import React, {useMemo} from 'react';
import {useSelector} from 'react-redux';

import {ITrainsCoach} from 'reducers/trains/order/types';

import passengersCountSelector from 'selectors/trains/order/passengersCountSelector';

import getCoachesGroupedByClasses from 'projects/trains/lib/order/coaches/getCoachesGroupedByClasses';
import {prepareTrainsFacilitiesAndAbilities} from './utilities/prepareTrainsFacilitiesAndAbilities';
import {clearTrainsLastClassDescriptionDot} from './utilities/clearTrainsLastClassDescriptionDot';
import {getTrainsCoachClassLabelWithCode} from './utilities/getTrainsCoachClassLabelWithCode';

import Flex from 'components/Flex/Flex';
import CoachFullAdditionalInfo from './components/CoachFullAdditionalInfo/CoachFullAdditionalInfo';
import CoachPreviewAdditionalInfo, {
    TCoachPreviewAdditionalInfoType,
} from './components/CoachPreviewAdditionalInfo/CoachPreviewAdditionalInfo';

interface ICoachAdditionalInfoProps {
    coach: ITrainsCoach;
    previewType: TCoachPreviewAdditionalInfoType;
}

const CoachAdditionalInfo: React.FC<ICoachAdditionalInfoProps> = props => {
    const {coach, previewType} = props;
    const passengers = useSelector(passengersCountSelector);
    const classes = getCoachesGroupedByClasses({coaches: [coach]});
    const activeClass = Object.values(classes)[0];
    const facilitiesAndAbilities = useMemo(() => {
        return prepareTrainsFacilitiesAndAbilities({
            activeClass,
            coach,
            passengers,
        });
    }, [activeClass, coach, passengers]);
    const description = useMemo(() => {
        if (!activeClass) {
            return null;
        }

        return clearTrainsLastClassDescriptionDot(activeClass.description);
    }, [activeClass]);
    const coachClassLabelWithCode = getTrainsCoachClassLabelWithCode(
        coach.serviceClass,
    );

    if (!activeClass) {
        return null;
    }

    const {owner} = activeClass;

    return (
        <Flex nowrap justifyContent="space-between" above={4}>
            <CoachPreviewAdditionalInfo
                activeClassOwner={owner}
                description={description}
                previewType={previewType}
                facilitiesAndAbilities={facilitiesAndAbilities}
                coachClassLabelWithCode={coachClassLabelWithCode}
            />
            {previewType === 'short' && (
                <CoachFullAdditionalInfo
                    coach={coach}
                    activeClassOwner={owner}
                    description={description}
                    facilitiesAndAbilities={facilitiesAndAbilities}
                    coachClassLabelWithCode={coachClassLabelWithCode}
                />
            )}
        </Flex>
    );
};

export default CoachAdditionalInfo;
