import React from 'react';

import {ITrainsCoach} from 'reducers/trains/order/types';
import {ECoachAbility} from 'types/trains/booking/ECoachAbility';
import {ECoachFacility} from 'types/trains/booking/ECoachFacility';

import {useBoolean} from 'utilities/hooks/useBoolean';
import {CHAR_SPACE} from 'utilities/strings/charCodes';

import * as i18nBlock from 'i18n/trains-coach-properties';

import Info from 'icons/16/Info';
import Box from 'components/Box/Box';
import Text from 'components/Text/Text';
import Heading from 'components/Heading/Heading';
import BottomSheet from 'components/BottomSheet/BottomSheet';

import CoachFacilitiesAndAbilities from '../CoachFacilitiesAndAbilities/CoachFacilitiesAndAbilities';

import cx from './CoachFullAdditionalInfo.scss';

interface ICoachFullAdditionalInfoProps {
    coach: ITrainsCoach;
    activeClassOwner: string;
    description: null | string;
    coachClassLabelWithCode: string;
    facilitiesAndAbilities: (ECoachFacility | ECoachAbility)[];
}

const CoachFullAdditionalInfo: React.FC<ICoachFullAdditionalInfoProps> =
    props => {
        const {
            coach,
            description,
            activeClassOwner,
            facilitiesAndAbilities,
            coachClassLabelWithCode,
        } = props;
        const {
            value: visibleBottomSheet,
            setFalse: hideBottomSheet,
            setTrue: showBottomSheet,
        } = useBoolean(false);
        const coachNumber = i18nBlock.coachDashNumberDashWithDashIsDashFirm({
            number: parseInt(coach.number, 10),
            isNFCoach: false,
        });

        return (
            <>
                <Info className={cx('infoIcon')} onClick={showBottomSheet} />
                <BottomSheet
                    isOpened={visibleBottomSheet}
                    onClose={hideBottomSheet}
                    contentClassName={cx('bottomSheetContent')}
                >
                    <Heading className={cx('title')} level={3}>
                        <span>{coachNumber}</span>
                        <span className={cx('coachClass')}>
                            {CHAR_SPACE}
                            {coachClassLabelWithCode}
                        </span>
                    </Heading>
                    {facilitiesAndAbilities.length > 0 && (
                        <Box between={2} below={4}>
                            <CoachFacilitiesAndAbilities
                                facilitiesAndAbilities={facilitiesAndAbilities}
                                itemSize="m"
                            />
                        </Box>
                    )}
                    {Boolean(description) && (
                        <Text className={cx('description')} tag="div">
                            {description}
                        </Text>
                    )}
                    <Text color="secondary" tag="div">
                        {`${i18nBlock.owner()}: ${activeClassOwner}`}
                    </Text>
                </BottomSheet>
            </>
        );
    };

export default CoachFullAdditionalInfo;
