import React from 'react';

import {ECoachAbility} from 'types/trains/booking/ECoachAbility';
import {ETrainsGoal} from 'utilities/metrika/types/goals/trains';
import {ECoachFacility} from 'types/trains/booking/ECoachFacility';

import {useReachGoal} from 'utilities/metrika/useReachGoal';

import * as i18nCommonBlock from 'i18n/common';
import * as i18nBlock from 'i18n/trains-coach-properties';

import Box from 'components/Box/Box';
import Flex from 'components/Flex/Flex';
import Text from 'components/Text/Text';
import Raskrivashka from 'components/Raskrivashka/Raskrivashka';

import CoachFacilitiesAndAbilities from '../CoachFacilitiesAndAbilities/CoachFacilitiesAndAbilities';

import cx from './CoachPreviewAdditionalInfo.scss';

export type TCoachPreviewAdditionalInfoType = 'full' | 'short';

interface ICoachPreviewAdditionalInfoProps {
    activeClassOwner: string;
    description: null | string;
    coachClassLabelWithCode: string;
    previewType: TCoachPreviewAdditionalInfoType;
    facilitiesAndAbilities: (ECoachFacility | ECoachAbility)[];
}

const PREVIEW_ICONS_COUNT = 4;

const CoachPreviewAdditionalInfo: React.FC<ICoachPreviewAdditionalInfoProps> =
    props => {
        const {
            description,
            facilitiesAndAbilities,
            activeClassOwner,
            previewType,
            coachClassLabelWithCode,
        } = props;
        const isFullView = previewType === 'full';
        const previewFacilitiesAndAbilities = isFullView
            ? facilitiesAndAbilities
            : facilitiesAndAbilities.slice(0, PREVIEW_ICONS_COUNT);
        const facilitiesAndAbilitiesNode =
            previewFacilitiesAndAbilities.length > 0 && (
                <Box className={cx('icons')} inline between={3}>
                    <CoachFacilitiesAndAbilities
                        facilitiesAndAbilities={previewFacilitiesAndAbilities}
                        itemClassName={cx('icon')}
                        itemSize={isFullView ? 'm' : 's'}
                    />
                </Box>
            );
        const descriptionNode = Boolean(description) && (
            <Text
                size="m"
                tag={isFullView ? 'div' : 'span'}
                color={isFullView ? 'secondary' : 'primary'}
                className={cx({description_hasOverflow: !isFullView})}
            >
                {description}
            </Text>
        );
        const handleToggleDescriptionView = useReachGoal(
            ETrainsGoal.PLACES_COACH_TOGGLE_DESCRIPTION_VIEW,
        );
        const descriptionWithAccordionNode = descriptionNode ? (
            <Raskrivashka
                theme="clear"
                label={
                    <Text color="secondary" size="m">
                        {i18nCommonBlock.moreDashInfo()}
                    </Text>
                }
                iconClassName={cx('descriptionIcon')}
                contentClassName={cx('descriptionContent')}
                onClickButton={handleToggleDescriptionView}
                contentIsInvolved={false}
            >
                {descriptionNode}
            </Raskrivashka>
        ) : (
            descriptionNode
        );

        const activeClassOwnerNode = (
            <Text size="m" tag={isFullView ? 'div' : 'span'}>
                {`${i18nBlock.owner()}: ${activeClassOwner}`}
            </Text>
        );

        if (isFullView) {
            return (
                <Flex className={cx('root')} justifyContent="space-between">
                    <Box between={1}>
                        {facilitiesAndAbilitiesNode}
                        {descriptionWithAccordionNode}
                        {activeClassOwnerNode}
                    </Box>
                    <Text size="m" color="secondary" whiteSpace="nowrap">
                        {coachClassLabelWithCode}
                    </Text>
                </Flex>
            );
        }

        if (facilitiesAndAbilitiesNode) {
            return facilitiesAndAbilitiesNode;
        }

        if (descriptionNode) {
            return descriptionNode;
        }

        return activeClassOwnerNode;
    };

export default CoachPreviewAdditionalInfo;
