import {useMemo, FC} from 'react';
import upperFirst from 'lodash/upperFirst';

import {PLACE_RESERVATION_TYPE} from 'projects/trains/constants/placeReservationType';

import {ITrainsCoach} from 'reducers/trains/order/types';
import {TTrainsStoreOrderSegment} from 'projects/trains/lib/segments/types';

import areCoachesAutoSeat from 'projects/trains/lib/order/coaches/areCoachesAutoSeat';
import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';
import getMaxPlacesReservationCountByType from 'projects/trains/lib/order/places/getMaxPlacesReservationCountByType';
import getPetsAttentionMessage from 'projects/trains/components/TrainsOrderPage/Coach/utilities/getPetsAttentionMessage';
import {getCoachArrivalLabelText} from 'projects/trains/components/TrainsOrderPage/Coach/components/CoachAttentions/utilities/getCoachArrivalLabelText';

import * as i18nCoachPropertiesBlock from 'i18n/trains-coach-properties';
import * as i18nTrainDetailsBlock from 'i18n/trains-train-details';

import Flex from 'components/Flex/Flex';
import Text from 'components/Text/Text';
import NotificationNewMobile from 'projects/trains/components/NotificationNewMobile/NotificationNewMobile';

interface ICoachAttentionsProps extends IWithQaAttributes {
    coach: ITrainsCoach;
    segment: TTrainsStoreOrderSegment;
    showNFCoach: boolean;
    showCoachArrival: boolean;
    withoutCoachNumberTitle: boolean;
}

interface IAttentionMessage {
    text: string;
    qaKey?: string;
}

const CoachAttentions: FC<ICoachAttentionsProps> = props => {
    const {
        coach: {placeReservationType},
        coach,
        segment,
        showNFCoach,
        showCoachArrival,
        withoutCoachNumberTitle,
    } = props;

    const attentions = useMemo<IAttentionMessage[]>(() => {
        const messages: IAttentionMessage[] = [];

        if (placeReservationType !== PLACE_RESERVATION_TYPE.USUAL) {
            messages.push({
                text: i18nCoachPropertiesBlock.wholeDashCompartment({
                    count: getMaxPlacesReservationCountByType(
                        placeReservationType,
                    ),
                }),
            });
        }

        if (areCoachesAutoSeat([coach])) {
            messages.push({
                text: i18nTrainDetailsBlock.autoDashSeatDashCoaches(),
                qaKey: 'autoSeatLabel',
            });
        }

        const petsAttentionMessage = getPetsAttentionMessage(coach);

        if (petsAttentionMessage) {
            messages.push({
                text: petsAttentionMessage,
            });
        }

        return messages;
    }, [coach, placeReservationType]);
    const nfCoachNode = useMemo(() => {
        if (!(showNFCoach && withoutCoachNumberTitle)) {
            return null;
        }

        return (
            <Text size="s-inset">
                {upperFirst(i18nCoachPropertiesBlock.nfDashCoach())}
            </Text>
        );
    }, [showNFCoach, withoutCoachNumberTitle]);
    const coachArrivalLabelNode = useMemo(() => {
        if (!showCoachArrival) {
            return null;
        }

        const {text} = getCoachArrivalLabelText({
            coach,
            segment,
            capitalize: true,
        });

        return (
            <NotificationNewMobile size="s-inset">{text}</NotificationNewMobile>
        );
    }, [coach, showCoachArrival, segment]);

    if (attentions.length === 0 && !coachArrivalLabelNode) {
        return null;
    }

    return (
        <Flex flexDirection="column" between={2}>
            {attentions.map(({text, qaKey}, index) => {
                const attentionQaAttributes = qaKey
                    ? prepareQaAttributes({
                          parent: props,
                          current: qaKey,
                      })
                    : undefined;

                return (
                    <Text key={index} size="s-inset" {...attentionQaAttributes}>
                        {text}
                    </Text>
                );
            })}
            {nfCoachNode}
            {coachArrivalLabelNode}
        </Flex>
    );
};

export default CoachAttentions;
