import {useCallback, FC} from 'react';
import {useDispatch} from 'react-redux';

import {IWithClassName} from 'types/withClassName';
import {ITrainsCoach, TrainsPassengersCount} from 'reducers/trains/order/types';

import {changeCoachAndResetRequirements} from 'reducers/trains/actions/changeCoachAndResetRequirements';

import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';
import areCoachesAutoSeat from 'projects/trains/lib/order/coaches/areCoachesAutoSeat';
import {getAutoSelectableCoach} from 'projects/trains/lib/order/getAutoSelectableCoach';

import * as i18nTrainDetailsBlock from 'i18n/trains-train-details';

import {IButtonProps} from 'components/Button/Button';
import TrainsOrderButton from 'projects/trains/components/TrainsOrderButton/TrainsOrderButton';

import cx from './CoachAutoSeatButton.scss';

interface ICoachAutoSeatButtonProps extends IWithClassName, IWithQaAttributes {
    coach: ITrainsCoach;
    passengers: TrainsPassengersCount;
}

const ORDER_BUTTON_PROPS: IButtonProps = {
    theme: 'secondary',
    size: 's',
};

const CoachAutoSeatButton: FC<ICoachAutoSeatButtonProps> = props => {
    const {coach, passengers, className} = props;
    const isAutoSeatCoach = areCoachesAutoSeat([coach]);
    const isSuitableCoach = Boolean(
        getAutoSelectableCoach([coach], passengers),
    );

    const dispatch = useDispatch();
    const handleButtonClick = useCallback(() => {
        dispatch(changeCoachAndResetRequirements(coach));
    }, [coach, dispatch]);

    if (!isAutoSeatCoach || !isSuitableCoach) {
        return null;
    }

    return (
        <TrainsOrderButton
            className={className}
            hintClassName={cx('hintWrapper')}
            hintText={i18nTrainDetailsBlock.autoDashSeatDashCoaches()}
            buttonProps={ORDER_BUTTON_PROPS}
            onClick={handleButtonClick}
            {...prepareQaAttributes(props)}
        />
    );
};

export default CoachAutoSeatButton;
