import React from 'react';
import upperFirst from 'lodash/upperFirst';

import {IWithClassName} from 'types/withClassName';
import {ITrainsCoach} from 'reducers/trains/order/types';

import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';
import {CHAR_NBSP} from 'utilities/strings/charCodes';

import * as i18nBlock from 'i18n/trains-coach-properties';

import Box from 'components/Box/Box';
import Text from 'components/Text/Text';

import cx from './CoachHeader.scss';

interface ICoachHeaderProps extends IWithClassName, IWithQaAttributes {
    coach: ITrainsCoach;
    showNFCoach: boolean;
    showCoachArrival: boolean;
    inline?: boolean;
    withoutCoachNumberTitle?: boolean;
}

const CoachHeader: React.FC<ICoachHeaderProps> = props => {
    const {
        className,
        coach,
        showNFCoach,
        inline,
        withoutCoachNumberTitle = false,
        showCoachArrival,
    } = props;

    if (withoutCoachNumberTitle && !showNFCoach && !showCoachArrival) {
        return null;
    }

    return (
        <Box
            between={1}
            className={cx('root', className, {
                root_inline: inline,
            })}
            {...prepareQaAttributes(props)}
        >
            {!withoutCoachNumberTitle && (
                <span
                    className={cx('coachNumber')}
                    {...prepareQaAttributes({
                        parent: props,
                        current: 'coachNumber',
                    })}
                >
                    {i18nBlock.coachDashNumberDashWithDashIsDashFirm({
                        number: parseInt(coach.number, 10),
                        isNFCoach: false,
                    })}
                </span>
            )}
            {showNFCoach && !withoutCoachNumberTitle && (
                <Text>
                    {CHAR_NBSP}
                    {upperFirst(i18nBlock.nfDashCoach())}
                </Text>
            )}
        </Box>
    );
};

export default CoachHeader;
