import {FC} from 'react';

import {GENDER_TYPE} from 'projects/trains/constants/genders';
import {ORDER_STEP} from 'projects/trains/constants/orderSteps';

import {ITrainsCoach, TrainsPassengersCount} from 'reducers/trains/order/types';
import {ITrainsSchema} from 'server/api/TrainsApi/types/ITrainsDetailsApiResponse';

import {deviceMods} from 'utilities/stylesUtils';
import getPlaceMap from 'projects/trains/lib/order/getPlaceMap';
import PriceScale from 'projects/trains/lib/order/priceScale/PriceScale';
import {useDeviceType} from 'utilities/hooks/useDeviceType';
import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';

import TransportSchema from 'projects/trains/components/TrainsOrderPage/TransportSchema/TransportSchema';

import cx from './CoachTransportSchema.scss';

interface ICoachTransportSchemaProps extends IWithQaAttributes {
    coach: ITrainsCoach;
    coachSchemaRenderingDelay: number;
    active: boolean;
    scale: PriceScale;
    schema: ITrainsSchema | null;
    gender: GENDER_TYPE | null;
    orderStep: ORDER_STEP;
    passengers: TrainsPassengersCount;
    orderPlaces: number[];
    schemeHasGroupedPlaces: boolean;
    schemaPlaceMaskPrefix: string;
    onCoachChange(): void;
    onSchemaCatchError?(): void;
}

const CoachTransportSchema: FC<ICoachTransportSchemaProps> = props => {
    const {
        coach,
        coachSchemaRenderingDelay,
        active,
        scale,
        schema,
        gender,
        orderStep,
        passengers,
        orderPlaces,
        schemeHasGroupedPlaces,
        schemaPlaceMaskPrefix,
        onCoachChange,
        onSchemaCatchError,
    } = props;

    const deviceType = useDeviceType();

    const placeMap = getPlaceMap({
        schema,
        coach,
        scale,
        orderPlaces,
        active,
        passengers,
        beddingOption: true,
        gender,
        schemeHasGroupedPlaces,
    });

    return (
        <section className={cx('root', deviceMods('root', deviceType))}>
            {coach.schemaId && schema && (
                <TransportSchema
                    legendClassName={cx('schemaLegend')}
                    deviceType={deviceType}
                    coachSchemaRenderingDelay={coachSchemaRenderingDelay}
                    coach={coach}
                    schema={schema}
                    schemeHasGroupedPlaces={schemeHasGroupedPlaces}
                    schemaPlaceMaskPrefix={schemaPlaceMaskPrefix}
                    gender={gender}
                    placeMap={placeMap}
                    orderStep={orderStep}
                    passengers={passengers}
                    orderPlaces={orderPlaces}
                    onCoachChange={onCoachChange}
                    scrollOffset={4}
                    onSchemaCatchError={onSchemaCatchError}
                    {...prepareQaAttributes({
                        parent: props,
                        current: 'transportSchema',
                    })}
                />
            )}
        </section>
    );
};

export default CoachTransportSchema;
