import {useEffect, useCallback, useRef} from 'react';
import {useInView} from 'react-intersection-observer';

import {TrainsPassengersCount} from 'reducers/trains/order/types';

import scrollToNode from 'utilities/dom/scrollToNode';
import {useExperiments} from 'utilities/hooks/useExperiments';
import {countPassengersWithPlaces as calculateCountPassengersWithPlaces} from 'projects/trains/lib/order/passengers/utils';

import {MODAL_ANIMATION_DURATION_MS} from 'components/Modal/Modal';
import {HideBodyVerticalScrollClassName} from 'components/HideBodyVerticalScroll/HideBodyVerticalScroll';

export interface IScrollToPassengersStepButtonProps {
    isMobile: boolean;
    orderPlaces: number[];
    passengers: TrainsPassengersCount;
}

const BOTTOM_SHIFT = 20;

export const useScrollToPassengersStepButton = ({
    isMobile,
    orderPlaces,
    passengers,
}: IScrollToPassengersStepButtonProps): ((
    node: HTMLElement | null,
) => void) => {
    const countOrderPlaces = orderPlaces.length;
    const joinedOrderPlaces = orderPlaces.join('_');
    const countPassengersWithPlaces =
        calculateCountPassengersWithPlaces(passengers);
    const {trainsScrollToPassengersStep} = useExperiments();
    const canUseScrollToButton = isMobile && trainsScrollToPassengersStep;
    const [inViewRef, hasIntersected] = useInView({
        triggerOnce: false,
        skip: !canUseScrollToButton,
    });
    const hasIntersectedRef = useRef<boolean>(false);
    const timeoutRef = useRef<number>();
    const buttonRef = useRef<HTMLElement | null>(null);
    const setButtonRef = useCallback(
        (node: HTMLElement | null) => {
            inViewRef(node);
            buttonRef.current = node;
        },
        [inViewRef, buttonRef],
    );

    hasIntersectedRef.current = hasIntersected;

    useEffect(() => {
        const isAllPlacesSelected =
            countOrderPlaces > 0 &&
            joinedOrderPlaces &&
            countOrderPlaces === countPassengersWithPlaces;

        if (
            isAllPlacesSelected &&
            canUseScrollToButton &&
            buttonRef.current &&
            !hasIntersectedRef.current
        ) {
            const {height} = buttonRef.current.getBoundingClientRect();
            const shiftY =
                height + BOTTOM_SHIFT - document.documentElement.clientHeight;
            const hasOpenedModal = document.documentElement.classList.contains(
                HideBodyVerticalScrollClassName,
            );

            clearTimeout(timeoutRef.current);

            timeoutRef.current = window.setTimeout(
                () => {
                    if (buttonRef.current) {
                        scrollToNode(buttonRef.current, {
                            shiftY,
                            behavior: 'smooth',
                        });
                    }
                },
                hasOpenedModal ? MODAL_ANIMATION_DURATION_MS : 0,
            );
        }
    }, [
        countOrderPlaces,
        countPassengersWithPlaces,
        canUseScrollToButton,
        joinedOrderPlaces,
    ]);

    useEffect(
        () => (): void => {
            clearTimeout(timeoutRef.current);
        },
        [],
    );

    return setButtonRef;
};
