import React from 'react';
import {useSelector} from 'react-redux';

import {IWithClassName} from 'types/withClassName';
import {ITrainsCoach} from 'reducers/trains/order/types';
import {ICoachesTab} from 'projects/trains/components/TrainsOrderPage/Coaches/types';

import currentTrainStateSelector from 'selectors/trains/order/currentTrainStateSelector';
import currentSegmentSelector from 'selectors/trains/order/currentSegmentSelector';
import currentTrainDetailsSelector from 'selectors/trains/order/currentTrainDetailsSelector';
import passengersCountSelector from 'selectors/trains/order/passengersCountSelector';
import currentCoachSelector from 'selectors/trains/order/currentCoachSelector';

import getCoachesInfo from 'projects/trains/lib/order/getCoachesInfo';
import getSchemaByCoach from 'projects/trains/lib/order/getSchemaByCoach';
import PriceScale from 'projects/trains/lib/order/priceScale/PriceScale';
import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';
import {useDeviceType} from 'utilities/hooks/useDeviceType';
import {deviceMods} from 'utilities/stylesUtils';
import getRepeatedCoachNumbers from 'projects/trains/components/TrainsOrderPage/Coaches/utilities/getRepeatedCoachNumbers';

import Coach from 'projects/trains/components/TrainsOrderPage/Coach/Coach';

import cx from './Coaches.scss';

const RENDERING_DELAY_GROUP_COUNT = 2;
const ADDITIONAL_DELAY_PER_GROUP = 100;

interface ICoachesProps extends IWithClassName, IWithQaAttributes {
    coaches: ITrainsCoach[];
    showCoachNumbers?: boolean;
}

const Coaches: React.FC<ICoachesProps> = props => {
    const {className, coaches, showCoachNumbers = true} = props;
    const passengers = useSelector(passengersCountSelector);
    const segment = useSelector(currentSegmentSelector);
    const trainDetails = useSelector(currentTrainDetailsSelector);
    const train = useSelector(currentTrainStateSelector);
    const gender = train?.gender ?? null;
    const coachType = train?.coachType ?? null;
    const coachCategory = train?.coachCategory ?? null;
    const orderPlaces = train?.orderPlaces ?? [];
    const activeCoach = useSelector(currentCoachSelector);
    const deviceType = useDeviceType();

    if (!trainDetails || !coachCategory || !coachType || !segment) {
        return null;
    }

    const {coaches: allCoaches, schemas, isFirm} = trainDetails;

    const scale = new PriceScale(coaches, true);
    const coachesInfo = getCoachesInfo(coaches);

    const showPrice = coachesInfo.prices.filter(Boolean).length !== 0;

    const arrivalTimes = new Set(allCoaches.map(coach => coach.arrival));
    const isDifferentCoachesArrival = arrivalTimes.size > 1;

    const repeatedCoachNumbers = getRepeatedCoachNumbers(coaches);

    const tabs: ICoachesTab[] = coaches.map((coach, index) => {
        const schema = getSchemaByCoach(coach, schemas);
        const coachSchemaRenderingDelay =
            Math.floor(index / RENDERING_DELAY_GROUP_COUNT) *
            ADDITIONAL_DELAY_PER_GROUP;

        return {
            active: Boolean(activeCoach?.index === coach.index),
            // Первые RENDERING_DELAY_GROUP_COUNT схемы отображаем синхронно,
            // остальные асинхронно, прибавляя по ADDITIONAL_DELAY_PER_GROUP мс каждой последующей группе
            coachSchemaRenderingDelay,
            coach,
            scale,
            schema,
            gender,
            segment,
            passengers,
            orderPlaces,
            showPrice,
            showNFCoach: isFirm && !coach.isFirm,
            withoutCoachNumberTitle: !showCoachNumbers,
            isDifferentCoachesArrival,
            schemaPlaceMaskPrefix: `${coach.number}_${
                repeatedCoachNumbers.has(coach.number) ? `${index}_` : ''
            }`,
        };
    });

    return (
        <section
            className={cx('root', deviceMods('root', deviceType), className)}
            {...prepareQaAttributes(props)}
        >
            <ul className={cx('items')} role="tablist">
                {tabs.map((tab, index) => (
                    <Coach
                        className={cx('tab')}
                        key={`tab-${tab.coach.index}`}
                        isLast={index === tabs.length - 1}
                        {...tab}
                        {...prepareQaAttributes({
                            parent: props,
                            current: 'coach',
                            key: tab.coach.number,
                        })}
                    />
                ))}
            </ul>
        </section>
    );
};

export default Coaches;
