import React from 'react';

import {IWithClassName} from 'types/withClassName';
import ITrainPassenger from 'server/api/GenericOrderApi/types/common/service/ITrainServiceInfo/ITrainPassenger/ITrainPassenger';

import {ITicketWithTrainInfo} from 'projects/trains/lib/order/getPassengerWithTicketsListByGenericOrderServices';
import {getTicketPlaces} from 'projects/trains/lib/order/places';
import {getTariffFromTicket} from 'projects/trains/lib/order/getTariffFromTicket';
import {useDeviceType} from 'utilities/hooks/useDeviceType';
import {deviceMods} from 'utilities/stylesUtils';
import {CHAR_EM_DASH, CHAR_NBSP} from 'utilities/strings/charCodes';
import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';
import getFromToByStations from 'projects/trains/lib/getFromToByStations';
import getTariffTextWithBaby from 'projects/trains/lib/order/tariffs/getTariffTextWithBaby';

import * as i18nCommonBlock from 'i18n/common';
import * as i18nTrainsOrderTicketsBlock from 'i18n/trains-order-tickets';

import Flex from 'components/Flex/Flex';
import Text from 'components/Text/Text';
import DotSeparator from 'components/DotSeparator/DotSeparator';
import PriceExplanation from 'projects/trains/components/TrainsOrderPage/PriceExplanation/PriceExplanation';
import Price from 'components/Price/Price';
import Intersperse from 'components/Intersperse/Intersperse';
import PassengerInfo from 'projects/trains/components/TrainsOrderPage/ConfirmStep/components/Passengers/components/Passenger/components/PassengerInfo/PassengerInfo';

import cx from './Passenger.scss';

interface IPassengerProps extends IWithClassName, IWithQaAttributes {
    passenger: ITrainPassenger;
    baby: ITrainPassenger | undefined;
    tickets: ITicketWithTrainInfo[];
    needPrintPlace: boolean;
    multipleTrains: boolean;
}

const Passenger: React.FC<IPassengerProps> = props => {
    const {
        className,
        passenger,
        baby,
        tickets,
        needPrintPlace,
        multipleTrains,
    } = props;

    const deviceType = useDeviceType();

    return (
        <Flex
            className={cx(className, 'root', deviceMods('root', deviceType))}
            flexDirection="column"
            {...prepareQaAttributes(props)}
        >
            <PassengerInfo
                passenger={passenger}
                {...prepareQaAttributes(props)}
            />

            {baby && (
                <PassengerInfo
                    className={cx('babyInfo')}
                    passenger={baby}
                    {...prepareQaAttributes({
                        parent: props,
                        current: 'baby',
                    })}
                />
            )}

            <Flex className={cx('tickets')} between={2} flexDirection="column">
                {tickets.map(({ticket, withBaby, trainInfo}, index) => {
                    if (!ticket) {
                        return null;
                    }

                    const {amount, payment} = ticket;
                    const tariff = getTariffTextWithBaby(
                        getTariffFromTicket(ticket),
                        withBaby,
                    );
                    const places = getTicketPlaces(ticket.places).join(', ');
                    const {stationFrom, stationTo} = trainInfo;
                    const {from, to} = getFromToByStations({
                        stationFrom,
                        stationTo,
                    });

                    const ticketQA = prepareQaAttributes({
                        parent: props,
                        current: 'ticket',
                        key: index,
                    });

                    return (
                        <Text
                            key={index}
                            size="s"
                            color="secondary"
                            className={cx('ticket')}
                            {...ticketQA}
                        >
                            <Intersperse separator={<DotSeparator />}>
                                {multipleTrains && (
                                    <span
                                        {...prepareQaAttributes({
                                            parent: ticketQA,
                                            current: 'direction',
                                        })}
                                    >
                                        {i18nCommonBlock.direction({from, to})}
                                    </span>
                                )}

                                {needPrintPlace && Boolean(places.length) && (
                                    <span
                                        {...prepareQaAttributes({
                                            parent: ticketQA,
                                            current: 'places',
                                        })}
                                    >
                                        {i18nTrainsOrderTicketsBlock.placeDashNumber(
                                            {
                                                numbers: places,
                                            },
                                        )}
                                    </span>
                                )}

                                <span
                                    className={cx('tariff')}
                                    {...prepareQaAttributes({
                                        parent: ticketQA,
                                        current: 'tariff',
                                    })}
                                >
                                    {tariff}
                                    {CHAR_NBSP}
                                    {CHAR_EM_DASH}{' '}
                                    {amount ? (
                                        <span
                                            className={cx(
                                                'priceAndExplanation',
                                            )}
                                        >
                                            <Price
                                                {...amount}
                                                {...prepareQaAttributes({
                                                    parent: ticketQA,
                                                    current: 'price',
                                                })}
                                            />

                                            {Boolean(payment.amount.value) && (
                                                <PriceExplanation
                                                    className={cx(
                                                        'explanation',
                                                    )}
                                                    ticketsPrice={
                                                        payment.amount
                                                    }
                                                    feePrice={payment.fee}
                                                    iconClassName={cx(
                                                        'infoIcon',
                                                    )}
                                                    {...prepareQaAttributes({
                                                        parent: ticketQA,
                                                        current:
                                                            'priceExplanation',
                                                    })}
                                                />
                                            )}
                                        </span>
                                    ) : (
                                        i18nTrainsOrderTicketsBlock.forDashFree()
                                    )}
                                </span>
                            </Intersperse>
                        </Text>
                    );
                })}
            </Flex>
        </Flex>
    );
};

export default Passenger;
