import {memo, FC} from 'react';
import moment from 'moment-timezone';

import {MAP_TRAINS_DOCUMENT_TYPES_TO_COMMON} from 'projects/trains/constants/documentTypes';

import ITrainPassenger from 'server/api/GenericOrderApi/types/common/service/ITrainServiceInfo/ITrainPassenger/ITrainPassenger';
import {IWithClassName} from 'types/withClassName';

import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';
import {getDocumentTitle} from 'projects/trains/lib/order/fields/utils';
import {formatDocumentNumber} from 'utilities/documents/formatDocumentNumber';
import {HUMAN_DATE_RU} from 'utilities/dateUtils/formats';
import {useDeviceType} from 'utilities/hooks/useDeviceType';
import {deviceMods} from 'utilities/stylesUtils';

import * as i18nTrainsOrderFieldsBlock from 'i18n/trains-order-fields';

import FieldLabel from 'components/FieldLabel/FieldLabel';
import Text from 'components/Text/Text';
import Flex from 'components/Flex/Flex';

import cx from './PassengerInfo.scss';

interface IPassengerInfoProps extends IWithClassName, IWithQaAttributes {
    passenger: ITrainPassenger;
}

const PassengerInfo: FC<IPassengerInfoProps> = props => {
    const {className, passenger} = props;

    const deviceType = useDeviceType();

    const {birthDate, docType, docId} = passenger;

    return (
        <Flex
            className={cx('root', deviceMods('root', deviceType), className)}
            inline={deviceType.isDesktop}
            between={deviceType.isMobile ? 4 : 2}
            justifyContent={
                deviceType.isMobile ? 'flex-start' : 'space-between'
            }
            flexDirection={deviceType.isMobile ? 'column' : 'row'}
        >
            <FieldLabel
                className={cx('field', 'field_lastName')}
                label={i18nTrainsOrderFieldsBlock.lastDashName()}
                {...prepareQaAttributes({
                    parent: props,
                    current: 'lastName',
                })}
            >
                <Text weight="bold">{passenger.lastName}</Text>
            </FieldLabel>

            <FieldLabel
                className={cx('field', 'field_firstName')}
                label={i18nTrainsOrderFieldsBlock.firstDashName()}
                {...prepareQaAttributes({
                    parent: props,
                    current: 'firstName',
                })}
            >
                {passenger.firstName}
            </FieldLabel>

            <FieldLabel
                className={cx('field', 'field_patronymic')}
                label={i18nTrainsOrderFieldsBlock.patronymic()}
                {...prepareQaAttributes({
                    parent: props,
                    current: 'patronymic',
                })}
            >
                {passenger.patronymic}
            </FieldLabel>

            {docId && (
                <FieldLabel
                    className={cx('field', 'field_documentNumber')}
                    label={getDocumentTitle(docType)}
                    {...prepareQaAttributes({
                        parent: props,
                        current: 'documentNumber',
                    })}
                >
                    {formatDocumentNumber(
                        docId,
                        MAP_TRAINS_DOCUMENT_TYPES_TO_COMMON[docType],
                    )}
                </FieldLabel>
            )}

            {birthDate && (
                <FieldLabel
                    className={cx('field', 'field_birthDate')}
                    label={i18nTrainsOrderFieldsBlock.birthDashDate()}
                    {...prepareQaAttributes({
                        parent: props,
                        current: 'birthDate',
                    })}
                >
                    {moment(birthDate).format(HUMAN_DATE_RU).toLowerCase()}
                </FieldLabel>
            )}
        </Flex>
    );
};

export default memo(PassengerInfo);
