import React from 'react';
import {useSelector} from 'react-redux';

import {IWithClassName} from 'types/withClassName';
import {ITrainsFilledSearchContext} from 'reducers/trains/context/types';
import {ITrainsDirectionAndIndex} from 'types/trains/common/segment/ITrainsDirectionAndIndex';

import {TrainsOrderStateInterface} from 'reducers/trains/order/reducer';

import changedPlacesSegmentsSelector from 'projects/trains/components/TrainsOrderPage/ConfirmStep/selectors/changedPlacesSegmentsSelector';

import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';
import {useDeviceType} from 'utilities/hooks/useDeviceType';
import {getTrainServices} from 'projects/trains/lib/order/getTrainServices';
import {deviceMods} from 'utilities/stylesUtils';

import * as i18nTrainsOrderConfirmBlock from 'i18n/trains-order-confirm';

import Flex from 'components/Flex/Flex';
import Heading from 'components/Heading/Heading';
import TrainPlaces from 'projects/trains/components/TrainsOrderPage/ConfirmStep/components/Places/components/TrainPlaces/TrainPlaces';
import PlacesNotifications from 'projects/trains/components/TrainsOrderPage/ConfirmStep/components/Places/components/PlacesNotifications/PlacesNotifications';
import Intersperse from 'components/Intersperse/Intersperse';
import Separator from 'components/Separator/Separator';

import cx from './Places.scss';

interface IPlacesProps extends IWithClassName, IWithQaAttributes {
    order: TrainsOrderStateInterface;
    context: ITrainsFilledSearchContext;

    goBackToChangePlaces(directionAndIndex?: ITrainsDirectionAndIndex): void;
}

const Places: React.FC<IPlacesProps> = props => {
    const {className, order, context, goBackToChangePlaces} = props;
    const {orderInfo} = order;

    const deviceType = useDeviceType();
    const changedPlacesSegments = useSelector(changedPlacesSegmentsSelector);

    if (!orderInfo) {
        return null;
    }

    const trainServices = getTrainServices(orderInfo);
    const multipleTrains = trainServices.length > 1;

    return (
        <Flex
            className={cx(className, 'root', deviceMods('root', deviceType))}
            flexDirection="column"
        >
            <PlacesNotifications
                showChangedPlacesNotification={changedPlacesSegments.length > 1}
                goBackToChangePlaces={goBackToChangePlaces}
            />

            <Heading
                className={cx('title')}
                level={deviceType.isDesktop ? 2 : 3}
            >
                {i18nTrainsOrderConfirmBlock.headerDashPlacesDashTitleDashRedesign()}
            </Heading>

            <Intersperse
                separator={<Separator className={cx('trainSeparator')} />}
            >
                {trainServices.map(trainService => (
                    <TrainPlaces
                        key={trainService.id}
                        trainService={trainService}
                        order={order}
                        context={context}
                        changedPlacesSegments={changedPlacesSegments}
                        multipleTrains={multipleTrains}
                        goBackToChangePlaces={goBackToChangePlaces}
                        {...prepareQaAttributes(
                            `${trainService.trainInfo.direction}-${trainService.trainInfo.segmentIndex}-trainsOrderPlaces`,
                        )}
                    />
                ))}
            </Intersperse>
        </Flex>
    );
};

export default Places;
