import React, {useMemo} from 'react';

import {IWithClassName} from 'types/withClassName';
import {ITrainGenericService} from 'server/api/GenericOrderApi/types/common/TGenericService';
import {ITrainsFilledSearchContext} from 'reducers/trains/context/types';
import {ITrainsDirectionAndIndex} from 'types/trains/common/segment/ITrainsDirectionAndIndex';

import {TrainsOrderStateInterface} from 'reducers/trains/order/reducer';

import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';
import {getPlacesWithDescription} from 'projects/trains/lib/order/places';
import {useDeviceType} from 'utilities/hooks/useDeviceType';
import {deviceMods} from 'utilities/stylesUtils';
import {hasPlacesWithPets} from 'projects/trains/lib/order/hasPlacesWithPets';
import {checkHasNonRefundablePassengerTariff} from 'projects/trains/lib/order/checkNonRefundablePassengerTariff';

import * as i18nTrainsOrderConfirmBlock from 'i18n/trains-order-confirm';

import Flex from 'components/Flex/Flex';
import WarningFilled from 'icons/16/WarningFilled';
import CoachInfo from 'projects/trains/components/TrainsOrderPage/ConfirmStep/components/Places/components/TrainPlaces/components/CoachInfo/CoachInfo';
import Schema from 'projects/trains/components/TrainsOrderPage/ConfirmStep/components/Places/components/TrainPlaces/components/Schema/Schema';
import TrainNotifications from 'projects/trains/components/TrainsOrderPage/ConfirmStep/components/Places/components/TrainPlaces/components/TrainNotifications/TrainNotifications';
import TextWithIcon from 'components/TextWithIcon/TextWithIcon';

import cx from './TrainPlaces.scss';

interface ITrainPlacesProps extends IWithClassName, IWithQaAttributes {
    trainService: ITrainGenericService;
    order: TrainsOrderStateInterface;
    context: ITrainsFilledSearchContext;
    changedPlacesSegments: ITrainsDirectionAndIndex[];
    multipleTrains: boolean;

    goBackToChangePlaces(directionAndIndex?: ITrainsDirectionAndIndex): void;
}

const TrainPlaces: React.FC<ITrainPlacesProps> = props => {
    const {
        className,
        trainService,
        order,
        context,
        changedPlacesSegments,
        multipleTrains,
        goBackToChangePlaces,
    } = props;
    const {direction, segmentIndex} = trainService.trainInfo;

    const deviceType = useDeviceType();

    const train = order.trains[direction][segmentIndex];
    const coach = train.coach ?? null;
    const eTicketStatus = Boolean(
        train.trainDetails?.trainDetails?.electronicTicket,
    );
    const places = getPlacesWithDescription(trainService.trainInfo.passengers);
    const hasNonRefundablePassengerTariff =
        checkHasNonRefundablePassengerTariff({
            passengers: trainService.trainInfo.passengers,
        });
    const petsAllowed = useMemo(
        () => hasPlacesWithPets(trainService),
        [trainService],
    );

    const showChangedPlacesNotification = useMemo(() => {
        if (!coach?.schemaId || changedPlacesSegments.length > 1) {
            return false;
        }

        const segment = changedPlacesSegments[0];

        if (!segment) {
            return false;
        }

        return (
            segment.direction === direction && segment.index === segmentIndex
        );
    }, [coach, changedPlacesSegments, direction, segmentIndex]);

    return (
        <div
            className={cx(className, 'root', deviceMods('root', deviceType))}
            {...prepareQaAttributes(props)}
        >
            <Flex
                className={cx('coachInfo')}
                flexDirection="column"
                between={1}
            >
                <CoachInfo
                    trainService={trainService}
                    coach={coach}
                    multipleTrains={multipleTrains}
                    {...prepareQaAttributes(props)}
                />
                <div
                    {...prepareQaAttributes({current: 'places', parent: props})}
                >
                    {i18nTrainsOrderConfirmBlock.placesDashTitle()}{' '}
                    {places.join(', ').toLowerCase()}
                </div>
                {hasNonRefundablePassengerTariff && (
                    <TextWithIcon
                        className={cx('nonRefundablePlace')}
                        iconLeftClassName={cx('nonRefundableIcon')}
                        iconLeft={WarningFilled}
                        text={i18nTrainsOrderConfirmBlock.nonRefundablePlace()}
                    />
                )}
            </Flex>

            <TrainNotifications
                className={cx('notifications')}
                context={context}
                direction={direction}
                index={segmentIndex}
                eTicketStatus={eTicketStatus}
                showChangedPlacesNotification={showChangedPlacesNotification}
                goBackToChangePlaces={goBackToChangePlaces}
            />

            <Schema
                className={cx('schema')}
                legendClassName={cx('schemaLegend')}
                trainService={trainService}
                order={order}
                {...prepareQaAttributes(props)}
            />

            <div className={cx('notes')}>
                {petsAllowed && (
                    <span>
                        {i18nTrainsOrderConfirmBlock.placesDashCoachDashPetsDashAllowed()}
                    </span>
                )}
            </div>
        </div>
    );
};

export default TrainPlaces;
