import React from 'react';
import upperFirst from 'lodash/upperFirst';

import {IWithClassName} from 'types/withClassName';
import {ITrainGenericService} from 'server/api/GenericOrderApi/types/common/TGenericService';
import {ITrainsCoach} from 'reducers/trains/order/types';

import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';
import {isGenderCompartment} from 'projects/trains/lib/order/placeCategories';
import getCoachTypeText from 'projects/trains/lib/getCoachTypeText';
import getCompartmentGenderText from 'projects/trains/lib/getCompartmentGenderText';
import getFromToByStations from 'projects/trains/lib/getFromToByStations';

import * as i18nTrainsOrderConfirmBlock from 'i18n/trains-order-confirm';
import * as i18nCommonBlock from 'i18n/common';

import Text from 'components/Text/Text';
import DotSeparator from 'components/DotSeparator/DotSeparator';

import {prepareCompartmentGender} from 'server/services/TrainsService/prepareData/prepareOrderResponse';

interface ICoachInfoProps extends IWithClassName, IWithQaAttributes {
    trainService: ITrainGenericService;
    coach: ITrainsCoach | null;
    multipleTrains: boolean;
}

const CoachInfo: React.FC<ICoachInfoProps> = props => {
    const {className, trainService, coach, multipleTrains} = props;

    const {carType, carNumber, compartmentGender, stationFrom, stationTo} =
        trainService.trainInfo;
    const compartmentIsGender = isGenderCompartment(compartmentGender);
    const {from, to} = getFromToByStations({stationFrom, stationTo});

    return (
        <Text className={className} weight="bold" size="m">
            {multipleTrains && (
                <div
                    {...prepareQaAttributes({
                        parent: props,
                        current: 'direction',
                    })}
                >
                    {i18nCommonBlock.direction({from, to})}
                </div>
            )}

            <span
                {...prepareQaAttributes({
                    parent: props,
                    current: 'coachNumber',
                })}
            >
                {i18nTrainsOrderConfirmBlock.coachDashNumber({
                    number: parseInt(carNumber, 10),
                })}
            </span>
            <DotSeparator />

            <span>{upperFirst(getCoachTypeText(carType))}</span>
            <DotSeparator />

            {compartmentIsGender && (
                <>
                    <span>
                        {upperFirst(
                            getCompartmentGenderText(
                                prepareCompartmentGender(compartmentGender),
                            ),
                        )}
                    </span>
                    <DotSeparator />
                </>
            )}

            {coach?.serviceClass.code && (
                <>
                    <span
                        {...prepareQaAttributes({
                            parent: props,
                            current: 'serviceClass',
                        })}
                    >
                        {upperFirst(
                            i18nTrainsOrderConfirmBlock.coachDashServiceDashClass(
                                {
                                    code: coach.serviceClass.code,
                                },
                            ),
                        )}
                    </span>
                    <DotSeparator />
                </>
            )}
        </Text>
    );
};

export default CoachInfo;
