import React from 'react';

import {IWithClassName} from 'types/withClassName';
import {ITrainGenericService} from 'server/api/GenericOrderApi/types/common/TGenericService';

import {TrainsOrderStateInterface} from 'reducers/trains/order/reducer';

import getSchemaByCoach from 'projects/trains/lib/order/getSchemaByCoach';
import {getActualPlaces} from 'projects/trains/lib/order/places';
import {useDeviceType} from 'utilities/hooks/useDeviceType';
import {getConfirmationPlaceMap} from 'projects/trains/lib/order/placeMap';
import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';

import TransportSchema from 'projects/trains/components/TrainsOrderPage/TransportSchema/TransportSchema';

interface ISchemaProps extends IWithClassName, IWithQaAttributes {
    legendClassName?: string;
    trainService: ITrainGenericService;
    order: TrainsOrderStateInterface;
}

const Schema: React.FC<ISchemaProps> = props => {
    const {
        className,
        legendClassName,
        trainService: {trainInfo},
        order: {passengers, orderInfo, trains},
    } = props;

    const deviceType = useDeviceType();

    const train = trains[trainInfo.direction][trainInfo.segmentIndex];

    if (!train) {
        return null;
    }

    const {coach, gender, segment, trainDetails, orderPlaces} = train;

    if (
        !segment ||
        !coach ||
        !coach.schemaId ||
        !orderInfo ||
        !trainDetails?.trainDetails
    ) {
        return null;
    }

    const schema = getSchemaByCoach(coach, trainDetails.trainDetails.schemas);

    if (!schema) {
        return null;
    }

    const reservedPlaces = getActualPlaces(trainInfo.passengers);

    return (
        <div
            className={className}
            {...prepareQaAttributes({
                parent: props,
                current: 'schemaWrapper',
            })}
        >
            <TransportSchema
                legendClassName={legendClassName}
                coach={coach}
                schemaPlaceMaskPrefix={`${coach.number}_`}
                schema={schema}
                gender={gender}
                deviceType={deviceType}
                placeMap={getConfirmationPlaceMap({
                    schema,
                    coach,
                    reservedPlaces,
                })}
                passengers={passengers}
                interactive={false}
                orderPlaces={orderPlaces}
                {...prepareQaAttributes({
                    parent: props,
                    current: 'transportSchema',
                })}
            />
        </div>
    );
};

export default Schema;
