import React from 'react';
import upperFirst from 'lodash/upperFirst';
import {useSelector} from 'react-redux';

import {IWithClassName} from 'types/withClassName';

import {TrainsOrderStateInterface} from 'reducers/trains/order/reducer';

import {IOrderTrainInfo} from 'selectors/trains/trainsInfoSelector';
import {trainsContextSelector} from 'selectors/trains/trainsContextSelector';

import {getTrainCompany} from './utilities/getTrainCompany';
import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';
import getTrainNumber from 'projects/trains/lib/order/getTrainNumber';
import {getTrainFirm} from './utilities/getTrainFirm';
import {useDeviceType} from 'utilities/hooks/useDeviceType';
import {deviceMods} from 'utilities/stylesUtils';
import {CHAR_EM_DASH, CHAR_NBSP} from 'utilities/strings/charCodes';
import {HUMAN_DATE_WITH_FULL_WEEKDAY} from 'utilities/dateUtils/formats';
import getFirstForwardTrainDetails from 'projects/trains/lib/complexOrder/getFirstForwardTrainDetails';
import getFirstForwardSegment from 'projects/trains/lib/complexOrder/getFirstForwardSegment';

import * as i18nTrainsTimezonesBlock from 'i18n/trains-timezones';
import * as i18nTrainsOrderConfirmBlock from 'i18n/trains-order-confirm';

import Box from 'components/Box/Box';
import Flex from 'components/Flex/Flex';
import Text from 'components/Text/Text';
import Label, {ELabelThemeType} from 'components/Label/Label';
import Link from 'components/Link/Link';
import SegmentTimeAndStation from 'projects/trains/components/SegmentTimeAndStation/SegmentTimeAndStation';
import DotSeparator from 'components/DotSeparator/DotSeparator';
import Separator from 'components/Separator/Separator';

import cx from './Train.scss';

interface ITrainProps extends IWithClassName, IWithQaAttributes {
    order: TrainsOrderStateInterface;
    trainInfo: IOrderTrainInfo | null;
}

const Train: React.FC<ITrainProps> = props => {
    const {className, order, trainInfo} = props;
    const trainDetails = getFirstForwardTrainDetails(order);
    const segment = getFirstForwardSegment(order);

    const deviceType = useDeviceType();
    const {from, to} = useSelector(trainsContextSelector);

    if (!segment || !trainDetails || !trainInfo) {
        return null;
    }

    const {isDesktop, isMobile} = deviceType;
    const {
        stationFrom,
        stationTo,
        departureMoment,
        arrivalMoment,
        eTicketStatus,
    } = trainInfo;

    const trainFirm = getTrainFirm(order);

    return (
        <div className={cx('wrapper')}>
            <Box
                className={cx(
                    className,
                    'root',
                    deviceMods('root', deviceType),
                )}
            >
                <div className={cx('title')}>
                    <Text
                        className={cx('title__cities')}
                        size={isDesktop ? 'xl' : 'l'}
                        weight="bold"
                    >
                        {`${from?.title}${CHAR_NBSP}${CHAR_EM_DASH} ${to?.title}`}
                    </Text>
                    <Text
                        className={cx('title__when')}
                        size={isDesktop ? 'xl' : 'm'}
                    >
                        {departureMoment.format(HUMAN_DATE_WITH_FULL_WEEKDAY)}
                    </Text>
                </div>

                {isMobile && <Separator className={cx('title-separator')} />}

                <Flex
                    className={cx('content')}
                    flexDirection={isDesktop ? 'row' : 'column'}
                    justifyContent="space-between"
                    between={isDesktop ? 0 : 3}
                    below="3"
                >
                    <Flex
                        className={cx('trainInfo')}
                        flexDirection={'column'}
                        between={3}
                    >
                        <Text
                            className={cx('company')}
                            size={'xl'}
                            {...prepareQaAttributes({
                                parent: props,
                                current: 'company',
                            })}
                        >
                            {getTrainCompany(order)}
                        </Text>

                        <div>
                            <div
                                className={cx('train')}
                                {...prepareQaAttributes({
                                    parent: props,
                                    current: 'trainTitle',
                                })}
                            >
                                {i18nTrainsOrderConfirmBlock.trainNumber({
                                    trainNumber: getTrainNumber(
                                        segment,
                                        trainDetails,
                                    ),
                                })}
                                {', '}
                                {segment.title}
                            </div>
                            <DotSeparator className={cx('dotSeparator')} />
                            {trainFirm && (
                                <div className={cx('firm')}>
                                    <Text
                                        color={'highlight'}
                                        {...prepareQaAttributes({
                                            parent: props,
                                            current: 'firm',
                                        })}
                                    >
                                        {upperFirst(trainFirm)}
                                    </Text>
                                </div>
                            )}
                        </div>
                    </Flex>

                    <SegmentTimeAndStation
                        className={cx('segment')}
                        departureDate={departureMoment}
                        arrivalDate={arrivalMoment}
                        departureStation={stationFrom}
                        arrivalStation={stationTo}
                        {...prepareQaAttributes(props)}
                    />
                </Flex>

                <Flex inline between={2} alignItems={'baseline'}>
                    <Label
                        className={cx('label')}
                        size={'m'}
                        theme={ELabelThemeType.ATTENTION}
                    >
                        {upperFirst(i18nTrainsTimezonesBlock.localDashTime())}
                    </Label>

                    {eTicketStatus ? (
                        <Label
                            className={cx('label')}
                            size={'m'}
                            theme={ELabelThemeType.NEUTRAL}
                        >
                            {i18nTrainsOrderConfirmBlock.eDashTicketDashBadge()}
                        </Label>
                    ) : (
                        <Link
                            className={cx('link')}
                            url="https://yandex.ru/support/travel/trains/e-registration.html"
                            target="_blank"
                            rel="nofollow noopener noreferrer"
                        >
                            {i18nTrainsOrderConfirmBlock.withoutDashErDashHelpDashLink()}
                        </Link>
                    )}
                </Flex>
            </Box>
        </div>
    );
};

export default Train;
