import {memo, FC, useState, useMemo} from 'react';
import {useDispatch, useSelector} from 'react-redux';

import {GENDER_TYPE} from 'projects/trains/constants/genders';

import {setPreferredGender} from 'reducers/trains/order/actions/trains';

import currentSegmentDirectionAndIndexSelector from 'selectors/trains/order/currentSegmentDirectionAndIndexSelector';

import useImmutableCallback from 'utilities/hooks/useImmutableCallback';
import getCompartmentGenderText from 'projects/trains/lib/getCompartmentGenderText';
import capitalizeFirstLetter from 'utilities/strings/capitalizeFirstLetter';

import * as i18n from 'i18n/trains-transport-schema';

import BottomSheet from 'components/BottomSheet/BottomSheet';
import {IGenderPopupProps} from 'projects/trains/components/TrainsOrderPage/GenderPopup/GenderPopup';
import Text from 'components/Text/Text';
import RadioButtonSlide from 'components/RadioButtonSlide/RadioButtonSlide';
import Button from 'components/Button/Button';

import cx from './GenderPopupMobile.scss';

interface IGenderPopupMobileProps extends IGenderPopupProps {}

const VARIANTS = [GENDER_TYPE.MIXED, GENDER_TYPE.FEMALE, GENDER_TYPE.MALE];

const GenderPopupMobile: FC<IGenderPopupMobileProps> = props => {
    const {visible, onClose, onChange} = props;
    const [gender, setGender] = useState(GENDER_TYPE.MIXED);
    const dispatch = useDispatch();

    const currentSegmentDirectionAndIndex = useSelector(
        currentSegmentDirectionAndIndexSelector,
    );

    const handleSelect = useImmutableCallback(() => {
        dispatch(
            setPreferredGender({
                ...currentSegmentDirectionAndIndex,
                data: gender,
            }),
        );
        onChange();
    });

    const variants = useMemo(
        () =>
            VARIANTS.map(variant => (
                <RadioButtonSlide.Option key={variant} value={variant}>
                    {capitalizeFirstLetter(getCompartmentGenderText(variant))}
                </RadioButtonSlide.Option>
            )),
        [],
    );

    return (
        <BottomSheet isOpened={visible} onClose={onClose}>
            <Text className={cx('title')} weight="bold" size="l">
                {i18n.selectDashGenderDashTitle()}
            </Text>

            <Text className={cx('description')}>
                {i18n.selectDashGenderDashText({link: false})}
            </Text>

            <RadioButtonSlide
                className={cx('genders')}
                value={gender}
                onChange={setGender as (v: string) => void}
            >
                {variants}
            </RadioButtonSlide>

            <Button
                className={cx('selectButton')}
                theme="primary"
                size="xl"
                width="max"
                onClick={handleSelect}
            >
                {i18n.selectGender()}
            </Button>
        </BottomSheet>
    );
};

export default memo(GenderPopupMobile);
