import React, {ChangeEvent, useCallback, useMemo} from 'react';
import noop from 'lodash/noop';
import {useDispatch, useSelector} from 'react-redux';

import {GENDER_TYPE} from 'projects/trains/constants/genders';

import {IWithClassName} from 'types/withClassName';

import {setPreferredGender} from 'reducers/trains/order/actions/trains';

import currentSegmentDirectionAndIndexSelector from 'selectors/trains/order/currentSegmentDirectionAndIndexSelector';

import capitalizeFirstLetter from 'utilities/strings/capitalizeFirstLetter';
import getCompartmentGenderText from 'projects/trains/lib/getCompartmentGenderText';
import {useDeviceType} from 'utilities/hooks/useDeviceType';

import RadioButton from 'components/RadioButton/RadioButton';

import cx from './GenderSelector.scss';

export interface IGenderSelectorProps extends IWithClassName {
    value: GENDER_TYPE | undefined;
    genders?: GENDER_TYPE[];
    // Блокировка срабатывания обработчиков по нажатию на кнопки
    disableMainAction?: boolean;
    onChange?(): void;
}

const BUTTONS = [GENDER_TYPE.MIXED, GENDER_TYPE.FEMALE, GENDER_TYPE.MALE];

const GenderSelector: React.FC<IGenderSelectorProps> = ({
    value,
    genders = BUTTONS,
    className,
    disableMainAction,
    onChange = noop,
}) => {
    const dispatch = useDispatch();
    const {isMobile} = useDeviceType();

    const currentSegmentDirectionAndIndex = useSelector(
        currentSegmentDirectionAndIndexSelector,
    );

    const handleChange = useCallback(
        (event: ChangeEvent<HTMLInputElement>) => {
            if (disableMainAction) {
                return;
            }

            const newValue = event.target.value as GENDER_TYPE;

            dispatch(
                setPreferredGender({
                    ...currentSegmentDirectionAndIndex,
                    data: newValue,
                }),
            );
            onChange();
        },
        [
            dispatch,
            onChange,
            disableMainAction,
            currentSegmentDirectionAndIndex,
        ],
    );

    const radioButtons = useMemo(
        () =>
            BUTTONS.map(item => ({
                value: item,
                disabled: !genders.includes(item),
                label: getCompartmentGenderText(item),
            })),
        [genders],
    );

    return (
        <RadioButton
            name="genderSelector"
            className={cx('root', className)}
            value={value}
            size={isMobile ? 'l' : 'm-inset'}
            onChange={handleChange}
            width="max"
        >
            {radioButtons.map(({label, value: buttonValue, disabled}) => (
                <RadioButton.Radio
                    key={buttonValue}
                    value={buttonValue}
                    checked={value === buttonValue}
                    disabled={disabled}
                >
                    {capitalizeFirstLetter(label)}
                </RadioButton.Radio>
            ))}
        </RadioButton>
    );
};

export default GenderSelector;
