import React from 'react';
import {useSelector} from 'react-redux';

import {ORDER_STEP} from 'projects/trains/constants/orderSteps';

import {IWithClassName} from 'types/withClassName';

import {orderStepSelector} from 'selectors/trains/order/orderStepSelector';
import passengersCountSelector from 'selectors/trains/order/passengersCountSelector';

import {insertJSXIntoKey} from 'utilities/tanker/insertJSXIntoKey';
import {countPassengersWithPlaces} from 'projects/trains/lib/order/passengers/utils';
import {CHAR_NBSP} from 'utilities/strings/charCodes';

import * as i18nBlockSummary from 'i18n/trains-order-summary';
import * as i18nBlockSummaryDetails from 'i18n/trains-order-summary-details';
import * as i18nBlockPartner from 'i18n/trains-train-partner-info';
import * as i18nBlockAgreement from 'i18n/trains-order-agreement';
import * as i18nBlockNotifications from 'i18n/trains-order-notifications';

import Link from 'components/Link/Link';

interface ILegalInformationProps extends IWithClassName {}

const LegalInformation: React.FC<ILegalInformationProps> = ({
    className,
}: ILegalInformationProps) => {
    const orderStep = useSelector(orderStepSelector);
    const passengers = useSelector(passengersCountSelector);

    return (
        <div className={className}>
            {orderStep === ORDER_STEP.CONFIRM ? '' : i18nBlockSummary.info()}

            {` ${i18nBlockSummaryDetails.priceDashYaDashHint({
                passengers: countPassengersWithPlaces(passengers),
            })} `}

            {orderStep === ORDER_STEP.PASSENGERS &&
                insertJSXIntoKey(i18nBlockAgreement.legalDashText)({
                    link: (
                        <Link
                            key="agreement-link"
                            theme="normal"
                            url={i18nBlockAgreement.linkDashUrl()}
                            target="_blank"
                            rel="noopener noreferrer"
                        >
                            {i18nBlockAgreement.linkDashText()}
                        </Link>
                    ),
                })}

            {` ${i18nBlockPartner.descriptionDashIm()}.`}

            {orderStep === ORDER_STEP.PASSENGERS && (
                <>
                    {CHAR_NBSP}
                    {insertJSXIntoKey(
                        i18nBlockNotifications.notificationsDashText,
                    )({
                        link: (
                            <Link
                                key="notifications-link"
                                theme="normal"
                                url={i18nBlockNotifications.linkDashUrl()}
                                target="_blank"
                                rel="noopener noreferrer"
                            >
                                {i18nBlockNotifications.linkDashText()}
                            </Link>
                        ),
                    })}
                </>
            )}
        </div>
    );
};

export default LegalInformation;
