import React, {useCallback, useMemo, RefObject} from 'react';

import EPopupDirection from 'components/Popup/types/EPopupDirection';

import {deviceMods} from 'utilities/stylesUtils';
import {useDeviceType} from 'utilities/hooks/useDeviceType';
import {REFUND_LINK} from 'projects/account/lib/orders/links';
import {insertJSXIntoKey} from 'utilities/tanker/insertJSXIntoKey';
import {prepareAndConvertPlacesRangeToString} from 'projects/trains/lib/order/places/convertPlacesToString';

import * as i18nBlock from 'i18n/trainsNonRefundableTariffSelector';

import Text from 'components/Text/Text';
import Link from 'components/Link/Link';
import {EMessageBoxPopupTheme} from 'components/MessageBoxPopup/MessageBoxPopup';
import MessageBoxPopupOrBottomSheet from 'components/MessageBoxPopupOrBottomSheet/MessageBoxPopupOrBottomSheet';

import cx from './NonRefundableTariffHelp.scss';

interface INonRefundableTariffHelpProps {
    places?: number[];
    isVisible: boolean;
    triggerRef: RefObject<HTMLElement>;
    onClose: () => void;
    discount?: number;
    hasRefundablePlace?: boolean;
}

const NonRefundableTariffHelp: React.FC<INonRefundableTariffHelpProps> =
    props => {
        const {
            places,
            onClose,
            isVisible,
            triggerRef,
            discount,
            hasRefundablePlace,
        } = props;
        const deviceType = useDeviceType();
        const messageBoxProps = useMemo(() => {
            return {
                anchorRef: triggerRef,
                theme: EMessageBoxPopupTheme.WHITE,
                direction: EPopupDirection.BOTTOM_RIGHT,
                secondaryOffset: 24,
            };
        }, [triggerRef]);

        const handleClose = useCallback(() => {
            onClose();
        }, [onClose]);

        if (!places || places.length === 0) {
            return null;
        }

        return (
            <MessageBoxPopupOrBottomSheet
                isVisible={isVisible}
                onClose={handleClose}
                messageBoxProps={messageBoxProps}
            >
                <div className={cx(deviceMods('root', deviceType))}>
                    <Text className={cx('title')} size="l" tag="div">
                        {i18nBlock.nonRefundableTariffLabel()}
                    </Text>
                    <Text className={cx('description')} size="m" tag="div">
                        {insertJSXIntoKey(
                            i18nBlock.nonRefundableFullTariffHelp,
                        )({
                            discount: discount
                                ? i18nBlock.dynamicDiscount({discount})
                                : '',
                            placesInfo: prepareAndConvertPlacesRangeToString({
                                places,
                            }),
                            link: (
                                <Link
                                    url={REFUND_LINK}
                                    target="_blank"
                                    rel="noopener noreferrer"
                                >
                                    {i18nBlock.helpLinkText()}
                                </Link>
                            ),
                        })}
                        {hasRefundablePlace && (
                            <>
                                <br />
                                {i18nBlock.refundablePlacesWithoutDiscountHelp()}
                            </>
                        )}
                    </Text>
                </div>
            </MessageBoxPopupOrBottomSheet>
        );
    };

export default NonRefundableTariffHelp;
