import React from 'react';
import {connect} from 'react-redux';
import {batchActions} from 'redux-batched-actions';
import {RouteComponentProps, withRouter} from 'react-router-dom';

import {EOrderErrorType} from 'projects/trains/constants/orderErrors';
import {ORDER_STEP} from 'projects/trains/constants/orderSteps';

import {IOrderError} from 'reducers/trains/order/types';

import {CustomDispatch} from 'reducers/trains/customDispatch';
import {IDevice} from 'reducers/common/commonReducerTypes';
import {StoreInterface} from 'reducers/storeTypes';
import {setContinueWithoutInsurance} from 'reducers/trains/order/actions/userChoice';
import {setOrderError} from 'reducers/trains/order/actions/view';
import {resetPayment} from 'reducers/trains/order/thunk/resetPayment';
import changeOrderStep from 'reducers/trains/order/thunk/changeOrderStep';
import choosePlacesAgain from 'reducers/trains/order/thunk/choosePlacesAgain';

import deviceTypeSelector from 'selectors/common/deviceTypeSelector';
import trainsOrderSelector from 'selectors/trains/order/trainsOrderSelector';

import InsuranceError from 'projects/trains/components/TrainsOrderPage/OrderError/InsuranceError';
import PaymentError from 'projects/trains/components/TrainsOrderPage/OrderError/PaymentError';
import PlacesError from 'projects/trains/components/TrainsOrderPage/OrderError/PlacesError';
import CommonError from 'projects/trains/components/TrainsOrderPage/OrderError/CommonError';
import ProcessExceptionStateError from 'projects/trains/components/TrainsOrderPage/OrderError/ProcessExceptionStateError/ProcessExceptionStateError';

function mapStateToProps(state: StoreInterface) {
    return {
        deviceType: deviceTypeSelector(state),
        error: trainsOrderSelector(state).error,
    };
}

function mapDispatchToProps(dispatch: CustomDispatch) {
    return {
        skipInsuranceError: (): void => {
            dispatch(
                batchActions([
                    setOrderError(null),
                    setContinueWithoutInsurance(true),
                ]),
            );
        },
        resetPayment: (): void => {
            dispatch(setOrderError(null));
            dispatch(resetPayment());
        },
        choosePassengers: (): void => {
            dispatch(setOrderError(null));
            dispatch(changeOrderStep({step: ORDER_STEP.PASSENGERS}));
        },
        choosePlaces: (): void => {
            dispatch(choosePlacesAgain());
        },
    };
}

export interface IOrderErrorPropTypes extends RouteComponentProps {
    deviceType: IDevice;
    error: IOrderError | null;
    skipInsuranceError: () => void;
    resetPayment: () => void;
    choosePassengers: () => void;
    choosePlaces: () => void;
}

const OrderError: React.FC<IOrderErrorPropTypes> = props => {
    const {error} = props;

    if (!error?.type) {
        return null;
    }

    if (error.type === EOrderErrorType.PLACES) {
        return <PlacesError {...props} />;
    }

    if (error.type === EOrderErrorType.INSURANCE) {
        return <InsuranceError {...props} />;
    }

    if (error.type === EOrderErrorType.PROCESS_EXCEPTION_STATE) {
        return <ProcessExceptionStateError {...props} />;
    }

    if (error.type === EOrderErrorType.PAYMENT) {
        return <PaymentError {...props} />;
    }

    return <CommonError {...props} />;
};

export default withRouter(
    connect(mapStateToProps, mapDispatchToProps)(OrderError),
);
