import React, {Component, ReactNode} from 'react';
import {Subtract} from 'utility-types';

import {IGenericOrderInfo} from 'server/api/GenericOrderApi/types/common/IGenericOrderInfo';

import {getPaymentTimeout} from 'projects/trains/lib/order/payment';

export interface IOrderTimerContainerProps {
    orderInfo: IGenericOrderInfo;
}

export interface IOrderTimerContainerState {
    timeLeft: number;
    isHighlighted: boolean;
}

export interface IBaseOrderTimerProps
    extends IOrderTimerContainerProps,
        IOrderTimerContainerState {}

/** TODO: move to containers */
function withTimer<P extends IBaseOrderTimerProps>(
    TimerComponent: React.ComponentType<P>,
) {
    return class OrderTimerContainer extends Component<
        Subtract<P, IOrderTimerContainerState>,
        IOrderTimerContainerState
    > {
        countdownTimeout?: NodeJS.Timeout;

        static defaultProps = {
            cancelOrderIsAvailable: true,
        };

        constructor(props: P) {
            super(props);

            const timeLeft = this.getTimeLeft();
            const isHighlighted = this.getHighlightStatus(timeLeft);

            this.state = {
                timeLeft,
                isHighlighted,
            };
        }

        componentDidMount(): void {
            this.updateTimeout();
        }

        componentWillUnmount(): void {
            if (this.countdownTimeout) {
                clearTimeout(this.countdownTimeout);
            }
        }

        getTimeLeft(): number {
            const timeLeft = getPaymentTimeout(this.props.orderInfo);

            return Math.max(timeLeft, 0);
        }

        getHighlightStatus(timeLeft: number): boolean {
            return timeLeft <= 60000 * 3;
        }

        updateTimeout(): void {
            this.countdownTimeout = setTimeout(() => {
                const timeLeft = this.getTimeLeft();
                const isHighlighted = this.getHighlightStatus(timeLeft);

                this.setState({
                    timeLeft,
                    isHighlighted,
                });

                if (timeLeft > 0) {
                    this.updateTimeout();
                }
            }, 1000);
        }

        render(): ReactNode {
            const {isHighlighted, timeLeft} = this.state;

            return (
                <TimerComponent
                    {...(this.props as P)}
                    timeLeft={timeLeft}
                    isHighlighted={isHighlighted}
                />
            );
        }
    };
}

export default withTimer;
