import React, {useCallback} from 'react';

import {PASSENGERS_TYPES} from 'projects/trains/constants/passengersTypes';

import {IWithClassName} from 'types/withClassName';
import {ITrainsTariffCategory} from 'server/api/TrainsApi/types/ITrainsDetailsApiResponse';
import {TrainsPassengersCount} from 'reducers/trains/order/types';
import {ETrainsGoal} from 'utilities/metrika/types/goals/trains';

import {
    getHint,
    getTitle,
    getMinMaxValues,
} from './utilities/prepareTitleAndHint';
import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';
import {reachGoal} from 'utilities/metrika';

import TravellersCount, {
    ITravellersCountProps,
} from 'components/TravellersCount/TravellersCount';

export interface IPassengersCountProps
    extends Pick<ITravellersCountProps, 'countWidth'>,
        IWithClassName,
        IWithQaAttributes {
    type: PASSENGERS_TYPES;
    passengers: TrainsPassengersCount;
    maxPlaces: number;
    tariffCategories?: ITrainsTariffCategory[];
    setPassengersCount: (type: PASSENGERS_TYPES, count: number) => void;
    disabled?: boolean;
    maxWidth?: boolean;
}

const PassengersCount: React.FC<IPassengersCountProps> = props => {
    const {
        className,
        type,
        passengers,
        maxWidth,
        maxPlaces,
        tariffCategories,
        setPassengersCount,
        disabled = false,
        countWidth,
    } = props;

    const onChange = useCallback(
        (value: number) => {
            setPassengersCount(type, value);
            reachGoal(ETrainsGoal.PASSENGERS_COUNT_CHANGED);
        },
        [setPassengersCount, type],
    );

    return (
        <div className={className} {...prepareQaAttributes(props)}>
            <TravellersCount
                countWidth={countWidth}
                value={passengers[type]}
                disabled={disabled}
                maxWidth={maxWidth}
                label={getTitle({
                    type,
                    tariffCategories,
                })}
                hint={getHint({
                    type,
                    tariffCategories,
                })}
                onChange={onChange}
                {...getMinMaxValues(type, passengers, maxPlaces)}
                {...prepareQaAttributes({
                    parent: props,
                    current: 'control',
                })}
            />
        </div>
    );
};

export default PassengersCount;
