import {PASSENGERS_TYPES} from 'projects/trains/constants/passengersTypes';
import {MAX_PASSENGERS_COUNT_IN_ORDER} from 'projects/trains/constants/order';

import {
    TrainsTariffCategoryCode,
    ITrainsTariffCategory,
} from 'server/api/TrainsApi/types/ITrainsDetailsApiResponse';
import {TrainsPassengersCount} from 'reducers/trains/order/types';

import {
    countPassengers,
    countPassengersWithPlaces,
} from 'projects/trains/lib/order/passengers/utils';
import {convertPassengerTypeToTariffType} from 'projects/trains/lib/convertPassengerTypeToTariffType';

import * as i18nBlock from 'i18n/trains-passengers-selector';

interface IMinMax {
    min: number;
    max: number;
}

export const getRemainingPassengers = (
    type: PASSENGERS_TYPES,
    passengers: TrainsPassengersCount,
) =>
    MAX_PASSENGERS_COUNT_IN_ORDER -
    countPassengers(passengers) +
    passengers[type];

export const getRemainingPlaces = (
    type: PASSENGERS_TYPES,
    passengers: TrainsPassengersCount,
    maxPlaces: number,
) => maxPlaces - countPassengersWithPlaces(passengers) + passengers[type];

export const getAdultsMinMax = (
    passengers: TrainsPassengersCount,
    maxPlaces: number,
): IMinMax => ({
    min: passengers[PASSENGERS_TYPES.BABIES],
    max: Math.min(
        getRemainingPassengers(PASSENGERS_TYPES.ADULTS, passengers),
        getRemainingPlaces(PASSENGERS_TYPES.ADULTS, passengers, maxPlaces),
    ),
});

export const getChildrenMinMax = (
    passengers: TrainsPassengersCount,
    maxPlaces: number,
): IMinMax => ({
    min: 0,
    max: Math.min(
        getRemainingPassengers(PASSENGERS_TYPES.CHILDREN, passengers),
        getRemainingPlaces(PASSENGERS_TYPES.CHILDREN, passengers, maxPlaces),
    ),
});

export const getBabiesMinMax = (
    passengers: TrainsPassengersCount,
): IMinMax => ({
    min: 0,
    max: Math.min(
        passengers[PASSENGERS_TYPES.ADULTS],
        getRemainingPassengers(PASSENGERS_TYPES.BABIES, passengers),
    ),
});

export function getMinMaxValues(
    type: PASSENGERS_TYPES,
    passengers: TrainsPassengersCount,
    maxPlaces: number,
): IMinMax {
    switch (type) {
        case PASSENGERS_TYPES.ADULTS:
            return getAdultsMinMax(passengers, maxPlaces);
        case PASSENGERS_TYPES.CHILDREN:
            return getChildrenMinMax(passengers, maxPlaces);
        case PASSENGERS_TYPES.BABIES:
            return getBabiesMinMax(passengers);
    }
}

export function getTitle({
    type,
    tariffCategories,
}: {
    type: PASSENGERS_TYPES;
    tariffCategories?: ITrainsTariffCategory[];
}): string {
    const maxAge = getMaxAge(
        tariffCategories,
        convertPassengerTypeToTariffType(type),
    );

    switch (type) {
        case PASSENGERS_TYPES.ADULTS: {
            return i18nBlock.adultsTitleNewMobile();
        }

        case PASSENGERS_TYPES.CHILDREN: {
            if (!maxAge) {
                return '';
            }

            return i18nBlock.childrenTitleNewMobile({maxAge});
        }

        case PASSENGERS_TYPES.BABIES: {
            if (!maxAge) {
                return '';
            }

            return i18nBlock.babiesTitleNewMobile({maxAge});
        }
    }
}

export function getHint({
    type,
    tariffCategories,
}: {
    type: PASSENGERS_TYPES;
    tariffCategories?: ITrainsTariffCategory[];
}): string {
    const minAge = getMinAge(
        tariffCategories,
        convertPassengerTypeToTariffType(type),
    );

    switch (type) {
        case PASSENGERS_TYPES.ADULTS: {
            if (!minAge) {
                return '';
            }

            return i18nBlock.adultsHintNewMobile({minAge});
        }

        case PASSENGERS_TYPES.CHILDREN: {
            return i18nBlock.childrenHintNewMobile();
        }

        case PASSENGERS_TYPES.BABIES: {
            return i18nBlock.babiesHintNewMobile();
        }
    }
}

function getMaxAge(
    tariffCategories: ITrainsTariffCategory[] | undefined,
    code: TrainsTariffCategoryCode,
): number | undefined {
    return tariffCategories?.find(category => category.code === code)?.maxAge;
}

function getMinAge(
    tariffCategories: ITrainsTariffCategory[] | undefined,
    code: TrainsTariffCategoryCode,
): number | undefined {
    return tariffCategories?.find(category => category.code === code)?.minAge;
}
