import React, {useCallback, useMemo} from 'react';
import {useDispatch, useSelector} from 'react-redux';

import {
    PASSENGERS_TYPES,
    PASSENGERS_TYPE_ORDER,
} from 'projects/trains/constants/passengersTypes';

import {IWithClassName} from 'types/withClassName';
import {EDirection} from 'types/common/EDirection';

import {setCount} from 'reducers/trains/order/actions/passengers';

import currentTrainDetailsSelector from 'selectors/trains/order/currentTrainDetailsSelector';
import currentSegmentDirectionAndIndexSelector from 'selectors/trains/order/currentSegmentDirectionAndIndexSelector';

import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';
import {convertPassengerTypeToTariffType} from 'projects/trains/lib/convertPassengerTypeToTariffType';

import PassengersCountSectionControls from 'projects/trains/components/TrainsOrderPage/PassengersCountSectionControls/PassengersCountSectionControls';

interface IPassengersCountSectionProps
    extends IWithClassName,
        IWithQaAttributes {}

const PassengersCountSection: React.FC<IPassengersCountSectionProps> =
    props => {
        const dispatch = useDispatch();
        const trainDetails = useSelector(currentTrainDetailsSelector);
        const {direction, index} = useSelector(
            currentSegmentDirectionAndIndexSelector,
        );
        const isDisabled = direction !== EDirection.FORWARD || index !== 0;

        const setPassengersCount = useCallback(
            (type: PASSENGERS_TYPES, count: number) => {
                dispatch(setCount[type](count));
            },
            [dispatch],
        );
        const passengersTypes = useMemo(() => {
            if (!trainDetails) {
                return [];
            }

            return PASSENGERS_TYPE_ORDER.filter(passengerType => {
                const tariffType =
                    convertPassengerTypeToTariffType(passengerType);

                return trainDetails.tariffCategories?.find(
                    category => category.code === tariffType,
                );
            });
        }, [trainDetails]);

        if (!trainDetails) {
            return null;
        }

        return (
            <PassengersCountSectionControls
                {...props}
                isDisabled={isDisabled}
                setPassengersCount={setPassengersCount}
                passengersTypes={passengersTypes}
                {...prepareQaAttributes(props)}
            />
        );
    };

export default PassengersCountSection;
