import React from 'react';
import {useSelector} from 'react-redux';

import {
    PASSENGERS_TYPES,
    PASSENGERS_TYPE_ORDER,
} from 'projects/trains/constants/passengersTypes';

import {IWithClassName} from 'types/withClassName';

import maxPlacesInCoachSelector from 'selectors/trains/order/maxPlacesInCoachSelector';
import passengersCountSelector from 'selectors/trains/order/passengersCountSelector';
import currentTrainDetailsSelector from 'selectors/trains/order/currentTrainDetailsSelector';

import {deviceMods} from 'utilities/stylesUtils';
import {useDeviceType} from 'utilities/hooks/useDeviceType';
import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';

import * as i18nBlock from 'i18n/trains-passengers-selector';

import Text from 'components/Text/Text';
import Flex from 'components/Flex/Flex';
import PassengersCount from 'projects/trains/components/TrainsOrderPage/PassengersCount/PassengersCount';
import PassengersCountNotification from 'projects/trains/components/TrainsOrderPage/PassengersCountNotification/PassengersCountNotification';

import cx from './PassengersCountSectionControls.scss';

interface IPassengersCountSectionControlsProps
    extends IWithClassName,
        IWithQaAttributes {
    passengersTypes: PASSENGERS_TYPES[];
    setPassengersCount: (type: PASSENGERS_TYPES, count: number) => void;
    isDisabled: boolean;
}

const PassengersCountSectionControls: React.FC<IPassengersCountSectionControlsProps> =
    props => {
        const {passengersTypes, setPassengersCount, isDisabled} = props;
        const deviceType = useDeviceType();
        const {isDesktop, isMobile} = deviceType;
        const passengers = useSelector(passengersCountSelector);
        const maxPlacesInCoach = useSelector(maxPlacesInCoachSelector);
        const trainDetails = useSelector(currentTrainDetailsSelector);

        if (!trainDetails) {
            return null;
        }

        const areAllPassengersTypes =
            passengersTypes.length === PASSENGERS_TYPE_ORDER.length;

        return (
            <section
                className={cx(deviceMods('root', deviceType))}
                {...prepareQaAttributes(props)}
            >
                <Text
                    className={cx('title')}
                    size={isDesktop ? 'xl' : 'l'}
                    tag="h2"
                    weight="bold"
                    {...prepareQaAttributes({
                        parent: props,
                        current: 'title',
                    })}
                >
                    {i18nBlock.title()}
                </Text>
                <Flex
                    flexDirection={isDesktop ? 'row' : 'column'}
                    justifyContent={
                        isDesktop && areAllPassengersTypes
                            ? 'space-between'
                            : 'flex-start'
                    }
                    between={isDesktop && areAllPassengersTypes ? 0 : 6}
                    inline={isDesktop}
                    above={isDesktop ? 4 : 3}
                >
                    {passengersTypes.map(type => (
                        <PassengersCount
                            key={type}
                            type={type}
                            maxWidth={isMobile}
                            countWidth={isDesktop ? 34 : 50}
                            maxPlaces={maxPlacesInCoach}
                            passengers={passengers}
                            tariffCategories={trainDetails.tariffCategories}
                            setPassengersCount={setPassengersCount}
                            disabled={isDisabled}
                            {...prepareQaAttributes({
                                parent: props,
                                current: 'passengersCountSelector',
                                key: type,
                            })}
                        />
                    ))}
                </Flex>
                <PassengersCountNotification />
            </section>
        );
    };

export default PassengersCountSectionControls;
