import React, {Component} from 'react';
import {connect} from 'react-redux';
import {FieldMetaState} from 'react-final-form';
import {RouteComponentProps, withRouter} from 'react-router-dom';
import {once, groupBy, times, isEmpty} from 'lodash';

import {MINUTE} from 'utilities/dateUtils/constants';
import {EFormKey} from 'constants/form/EFormKey';
import {mapTrainsGenderToCommon} from 'projects/trains/constants/genders';
import {PASSENGERS_TYPES} from 'projects/trains/constants/passengersTypes';
import {validationInfo} from 'projects/trains/constants/booking/validation/validationInfo';
import {EDocumentType} from 'constants/document/documentTypes';
import {RU_CITIZENSHIP_CODE2} from 'constants/document/citizenship';
import {MAP_COMMON_DOCUMENT_TYPES_TO_TRAINS} from 'projects/trains/constants/documentTypes';

import {
    EFieldName,
    EGroupName,
    IPassenger,
} from 'types/trains/booking/ITrainsBookFormValues';
import {ETrainsGoal} from 'utilities/metrika/types/goals/trains';
import {IDictionary} from 'types/IDictionary';

import requestCountries from 'reducers/trains/order/thunk/requestCountries';
import changeOrderStepAction from 'reducers/trains/order/thunk/changeOrderStep';

import dataSelector from 'selectors/trains/order/passengers/passengersOrderStep';

import {reachGoal, params} from 'utilities/metrika';
import {reloadPage} from 'utilities/browser/reloadPage';
import {prepareQaAttributes} from 'utilities/qaAttributes/qaAttributes';
import {getBookingPassengers} from 'utilities/localStorage/booking/getBookingPassengers';
import {FIRST_PLACES_STEP} from 'projects/trains/lib/order/steps/firstPlacesStep';
import {getStringError} from 'components/FormField/utilities/getValidationErrorMessage';
import {getAllowedDocumentTypes} from 'projects/trains/lib/order/passengers/documentTypes';
import {omitDocumentByValidation} from 'components/BookingPassengerForm/components/DocumentsContext/utilities/omitDocumentsByValidation';

import * as i18nBlock from 'i18n/trains-error-page-order';

import Form from 'components/Form/Form';
import Separator from 'components/Separator/Separator';
import Intersperse from 'components/Intersperse/Intersperse';
import BookingLayout from 'components/Layouts/BookingLayout/BookingLayout';
import UpdateNotification from 'projects/trains/components/UpdateNotification/UpdateNotification';
import ContactsForm from './components/ContactsForm/ContactsForm';
import PassengerForm from './components/PassengerForm/PassengerForm';
import FormDataAdapter from './components/FormDataAdapter/FormDataAdapter';

import cx from './TrainsOrderPassengersStep.scss';

const PASSENGERS_STEP_QA = 'passengersStep';

interface IGroupedPassenger {
    ageGroup: PASSENGERS_TYPES;
    groupIndex: number;
}

const mapDispatchToProps = {
    // data
    requestCountries,
    changeOrderStep: changeOrderStepAction,
};

type TProps = ReturnType<typeof dataSelector> &
    typeof mapDispatchToProps &
    RouteComponentProps;

class TrainsOrderPassengersStep extends Component<TProps> {
    private static handleShowUpdateNotification(): void {
        reachGoal(ETrainsGoal.PASSENGERS_EXPIRED);
    }

    private static createGroupedPassenger(
        ageGroup: PASSENGERS_TYPES,
        index: number,
    ): IGroupedPassenger {
        return {
            groupIndex: index,
            ageGroup,
        };
    }

    private getStoragePassengers = once(() => {
        const {userInfo, orderGender} = this.props;
        const coachCommonGender = mapTrainsGenderToCommon(orderGender);
        const storagePassengers = getBookingPassengers<IPassenger>(
            EFormKey.TRAIN_BOOK,
            userInfo,
        ).map(passenger => omitDocumentByValidation(passenger, validationInfo));

        const filteredPassengers = coachCommonGender
            ? storagePassengers.filter(
                  passenger =>
                      passenger.ageGroup !== PASSENGERS_TYPES.ADULTS ||
                      passenger.sex === coachCommonGender,
              )
            : storagePassengers;

        return groupBy(filteredPassengers, ({ageGroup}) => ageGroup);
    });

    componentDidMount(): void {
        this.props.requestCountries();

        reachGoal(ETrainsGoal.ORDER_PASSENGERS_PAGE_LOAD);
    }

    private getInitialPassengerData = (
        storagePassengers: IDictionary<IPassenger[]>,
        ageGroup: PASSENGERS_TYPES,
        groupIndex: number,
    ): IPassenger | undefined => {
        const {
            allowedDocumentTypes,
            userInfo: {citizenship},
        } = this.props;

        if (!isEmpty(storagePassengers)) {
            return storagePassengers[ageGroup]?.[groupIndex];
        }

        if (
            citizenship &&
            citizenship !== RU_CITIZENSHIP_CODE2 &&
            getAllowedDocumentTypes(ageGroup, allowedDocumentTypes).includes(
                MAP_COMMON_DOCUMENT_TYPES_TO_TRAINS[EDocumentType.OTHER],
            )
        ) {
            return {
                [EFieldName.ageGroup]: ageGroup,
                citizenship,
                documentType: EDocumentType.OTHER,
            };
        }
    };

    private onUpdatePage = (): void => {
        const {changeOrderStep} = this.props;

        changeOrderStep(FIRST_PLACES_STEP);
        reloadPage();
        reachGoal(ETrainsGoal.PASSENGERS_EXPIRED_REFRESH_CLICK);
    };

    private onUpdateNotificationSkip = (): void => {
        reachGoal(ETrainsGoal.PASSENGERS_EXPIRED_CONTINUE);
    };

    private handleErrorShow = (
        fieldName: string,
        meta: FieldMetaState<string>,
    ): void => {
        const {error, submitError} = meta;
        const stringBlurError =
            getStringError(error, false) || getStringError(submitError, false);

        if (stringBlurError && meta.dirty) {
            params({trains: {passengersStep: {blurError: fieldName}}});
        }
    };

    private renderPassengersNewForm(): React.ReactNode {
        const {
            seats,
            countries,
            deviceType,
            passengersWithDocuments,
            deviceType: {isMobile},
            passengers,
            tariffCategories,
            allowedDocumentTypes,
            isCppk,
            lastSegmentDeparture,
        } = this.props;
        const storagePassengers = this.getStoragePassengers();
        const passengerAgeGroup = [
            ...times(passengers.adults, index =>
                TrainsOrderPassengersStep.createGroupedPassenger(
                    PASSENGERS_TYPES.ADULTS,
                    index,
                ),
            ),
            ...times(passengers.children, index =>
                TrainsOrderPassengersStep.createGroupedPassenger(
                    PASSENGERS_TYPES.CHILDREN,
                    index,
                ),
            ),
            ...times(passengers.babies, index =>
                TrainsOrderPassengersStep.createGroupedPassenger(
                    PASSENGERS_TYPES.BABIES,
                    index,
                ),
            ),
        ];

        const Container = isMobile
            ? BookingLayout.Card
            : BookingLayout.Card.Section;

        return passengerAgeGroup.map(({ageGroup, groupIndex}, index) => (
            <Form.FieldGroup
                key={`${ageGroup}.${groupIndex}`}
                groupId={`${ageGroup}.${groupIndex}`}
            >
                <Container>
                    <PassengerForm
                        seats={seats[index]}
                        passengersWithDocuments={passengersWithDocuments[index]}
                        deviceType={deviceType}
                        ageGroup={ageGroup}
                        passengerIndex={index}
                        ageGroupIndex={groupIndex}
                        countries={countries}
                        initialValues={this.getInitialPassengerData(
                            storagePassengers,
                            ageGroup,
                            groupIndex,
                        )}
                        tariffCategories={tariffCategories}
                        allowedDocumentTypes={allowedDocumentTypes}
                        isCppk={isCppk}
                        lastSegmentDeparture={lastSegmentDeparture}
                        onChangeError={this.handleErrorShow}
                        {...prepareQaAttributes({
                            key: String(index),
                            parent: PASSENGERS_STEP_QA,
                            current: 'passengerSection',
                        })}
                    />
                </Container>
            </Form.FieldGroup>
        ));
    }

    private renderContactsForm(): React.ReactNode {
        return (
            <ContactsForm
                onChangeError={this.handleErrorShow}
                {...prepareQaAttributes({
                    parent: PASSENGERS_STEP_QA,
                    current: 'contacts',
                })}
            />
        );
    }

    private renderContent(): React.ReactNode {
        const {deviceType} = this.props;

        if (deviceType.isMobile) {
            return (
                <>
                    <Form.FieldGroup groupId={`${EGroupName.passengers}`}>
                        {this.renderPassengersNewForm()}
                    </Form.FieldGroup>

                    <BookingLayout.Card>
                        {this.renderContactsForm()}
                    </BookingLayout.Card>
                </>
            );
        }

        return (
            <>
                <BookingLayout.Card sectioned>
                    <div className={cx('passengers')}>
                        <Form.FieldGroup groupId={`${EGroupName.passengers}`}>
                            <Intersperse
                                separator={
                                    <Separator className={cx('separator')} />
                                }
                            >
                                {this.renderPassengersNewForm()}
                            </Intersperse>
                        </Form.FieldGroup>
                    </div>
                </BookingLayout.Card>

                <BookingLayout.Card>
                    {this.renderContactsForm()}
                </BookingLayout.Card>
            </>
        );
    }

    render(): React.ReactNode {
        return (
            <>
                <BookingLayout.Forms>
                    {this.renderContent()}
                </BookingLayout.Forms>

                <UpdateNotification
                    timeout={30 * MINUTE}
                    primaryActionText={i18nBlock.buttonDashToDashPlacesDashStep()}
                    onUpdate={this.onUpdatePage}
                    onSkip={this.onUpdateNotificationSkip}
                    canReloadPageAfterAction={false}
                    onShowNotification={
                        TrainsOrderPassengersStep.handleShowUpdateNotification
                    }
                />
                <FormDataAdapter />
            </>
        );
    }
}

export default connect(
    dataSelector,
    mapDispatchToProps,
)(withRouter(TrainsOrderPassengersStep));
