import React, {useMemo, useState} from 'react';
import {FieldMetaState} from 'react-final-form';
import {times} from 'lodash';
import {useSelector} from 'react-redux';

import {EFormKey} from 'constants/form/EFormKey';
import {PASSENGERS_TYPES} from 'projects/trains/constants/passengersTypes';

import {IWithClassName} from 'types/withClassName';
import {
    EGroupName,
    IFormContacts,
} from 'types/trains/booking/ITrainsBookFormValues';

import passengersCountSelector from 'selectors/trains/order/passengersCountSelector';

import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';
import {deviceMods} from 'utilities/stylesUtils';
import {getBookingContacts} from 'utilities/localStorage/booking/getBookingContacts';
import {getEmailValidationError} from 'components/BookingContactsForm/utilities/getEmailValidationError';
import {useUserInfo} from 'utilities/hooks/useUserInfo';
import {useDeviceType} from 'utilities/hooks/useDeviceType';

import * as i18nBlock from 'i18n/trains-order-contacts';

import Form from 'components/Form/Form';
import Separator from 'components/Separator/Separator';
import BookingContactsForm from 'components/BookingContactsForm/BookingContactsForm';
import EmailOrPhoneInput from './components/EmailOrPhoneInput/EmailOrPhoneInput';
import AdditionalContactsCheckbox from './components/AdditionalContactsCheckbox/AdditionalContactsCheckbox';

import cx from './ContactsForm.scss';

interface IContactsFormProps extends IWithQaAttributes, IWithClassName {
    onChangeError: (fieldName: string, meta: FieldMetaState<string>) => void;
}

const VALIDATION = {
    email: getEmailValidationError,
};

const ContactsForm: React.FC<IContactsFormProps> = ({
    className,
    onChangeError,
    ...props
}) => {
    const deviceType = useDeviceType();
    const userInfo = useUserInfo();
    const {adults: adultsCount} = useSelector(passengersCountSelector);

    const storageOrPassportContacts = useMemo(
        () => getBookingContacts<IFormContacts>(EFormKey.TRAIN_BOOK, userInfo),
        [userInfo],
    );

    const initialCheckbox =
        storageOrPassportContacts.useOnlyMainContacts ?? true;

    const [isCheckedAdditionalContacts, setUseAdditionalContacts] =
        useState(initialCheckbox);

    return (
        <section
            className={cx(className, 'root', deviceMods('root', deviceType))}
            {...prepareQaAttributes(props)}
        >
            <Form.FieldGroup groupId={EGroupName.contacts}>
                <BookingContactsForm
                    description={i18nBlock.attention()}
                    fieldsValidation={VALIDATION}
                    initialValues={storageOrPassportContacts}
                    onChangeError={onChangeError}
                    {...prepareQaAttributes(props)}
                />
            </Form.FieldGroup>

            {adultsCount > 1 && (
                <>
                    <AdditionalContactsCheckbox
                        className={cx('additionContactsCheckbox')}
                        initialValue={initialCheckbox}
                        onChange={setUseAdditionalContacts}
                    />

                    {!isCheckedAdditionalContacts && (
                        <Form.FieldGroup
                            groupId={`${EGroupName.passengers}.${PASSENGERS_TYPES.ADULTS}`}
                        >
                            <Separator className={cx('separator')} />

                            {times(adultsCount).map((_, index) => (
                                <Form.FieldGroup key={index} groupId={index}>
                                    <EmailOrPhoneInput
                                        className={cx('formControl')}
                                        deviceType={deviceType}
                                        passengerIndex={index}
                                        onChangeError={onChangeError}
                                        {...prepareQaAttributes({
                                            parent: props,
                                            current: 'emailOrPhone',
                                        })}
                                    />
                                </Form.FieldGroup>
                            ))}
                        </Form.FieldGroup>
                    )}
                </>
            )}
        </section>
    );
};

export default ContactsForm;
