import React, {ReactNode, useMemo} from 'react';
import {FieldMetaState} from 'react-final-form';

import {IWithDeviceType} from 'types/withDeviceType';
import {IWithClassName} from 'types/withClassName';
import {
    EFieldName,
    IPassenger,
} from 'types/trains/booking/ITrainsBookFormValues';

import {deviceMods} from 'utilities/stylesUtils';
import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';
import getPassengerFullName from './utilities/getPassengerFullName';

import * as i18nBlock from 'i18n/trains-order-contacts';

import Input from 'components/Input/Input';
import FormField from 'components/FormField/FormField';
import {useFieldValue} from 'components/Form/Form';
import useFillTravelerContacts from 'projects/trains/components/TrainsOrderPage/PassengersStep/components/ContactsForm/components/EmailOrPhoneInput/hooks/useFillTravelerContacts/useFillTravelerContacts';

import cx from './EmailOrPhoneInput.scss';

interface IEmailOrPhoneInputProps
    extends IWithClassName,
        IWithDeviceType,
        IWithQaAttributes {
    passengerIndex: number;
    onChangeError: (fieldName: string, meta: FieldMetaState<string>) => void;
}

const EmailOrPhoneInput: React.FC<IEmailOrPhoneInputProps> = ({
    className,
    deviceType,
    passengerIndex,
    onChangeError,
    ...rest
}) => {
    const passenger = useFieldValue<IPassenger>();

    const title = useMemo(() => {
        return getPassengerFullName(passengerIndex, passenger);
    }, [passenger, passengerIndex]);

    useFillTravelerContacts(passenger);

    return (
        <div className={cx(deviceMods('root', deviceType), className)}>
            <div className={cx('passengerName')}>{title}</div>

            <FormField
                name={EFieldName.additionalContact}
                title={i18nBlock.emailOrPhone()}
                deviceType={deviceType}
                onChangeError={onChangeError}
                control={(
                    {input},
                    {error, inputRef, controlRef},
                ): ReactNode => (
                    <Input
                        {...input}
                        id={input.name}
                        inputRef={(currentInputRef): void => {
                            inputRef.current = currentInputRef;
                        }}
                        innerRef={controlRef}
                        size="l"
                        state={error ? 'error' : undefined}
                        type="text"
                        {...prepareQaAttributes({
                            parent: rest,
                            current: 'contacts-emailOrPhone',
                        })}
                    />
                )}
            />
        </div>
    );
};

export default EmailOrPhoneInput;
