import {PureComponent, ReactNode} from 'react';
import {FieldMetaState} from 'react-final-form';
import moment from 'moment-timezone';

import {
    MAP_TRAINS_DOCUMENT_TYPES_TO_COMMON,
    TRAINS_DOCUMENT_TYPES,
} from 'projects/trains/constants/documentTypes';
import {PASSENGERS_TYPES} from 'projects/trains/constants/passengersTypes';
import {validationInfo} from 'projects/trains/constants/booking/validation/validationInfo';
import {EDocumentType} from 'constants/document/documentTypes';

import {IWithDeviceType} from 'types/withDeviceType';
import {ITrainCountrySelect} from 'reducers/trains/order/types';
import {IPassengerWithDocumentsAndBonusCardsDTO} from 'server/api/TravelersApi/types/IPassengerDTO';
import {
    EFieldName,
    IPassenger,
    ITrainsBookFormValues,
} from 'types/trains/booking/ITrainsBookFormValues';
import {ITrainsTariffCategory} from 'server/api/TrainsApi/types/ITrainsDetailsApiResponse';

import {deviceModDesktop} from 'utilities/stylesUtils';
import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';
import {HUMAN_DATE_RU} from 'utilities/dateUtils/formats';
import getPassengerAge from 'projects/trains/lib/order/passengers/getPassengerAge';
import {getPassengerTypeTitle} from 'projects/trains/lib/order/getPassengerTypeTitle';
import {getDocumentList} from './utilities/getDocumentList';
import {validateBirthdate} from './utilities/validateBirthDate';
import {validateDocumentNumber} from './utilities/validateDocumentNumber';

import * as tariffKeyset from 'i18n/trains-order-summary-details';
import {babiesDashWithDashAdultDashOverDashTenDashYearsDashOld} from 'i18n/trains-passengers-selector';

import BookingPassengerForm from 'components/BookingPassengerForm/BookingPassengerForm';
import AdditionFieldsFooter from './components/AdditionFieldsFooter/AdditionFieldsFooter';
import PassengerDataAdapter from './components/PassengerDataAdapter/PassengerDataAdapter';

import cx from './PassengerForm.scss';

const DOCUMENT_TYPES_WITH_AGE_LIMIT = [
    EDocumentType.RU_NATIONAL_PASSPORT,
    EDocumentType.RU_SEAMAN_PASSPORT,
    EDocumentType.RU_MILITARY_ID,
];

export interface IPassengerFormProps
    extends IWithDeviceType,
        IWithQaAttributes {
    passengerIndex: number;
    ageGroupIndex: number;
    ageGroup: PASSENGERS_TYPES;
    passengersWithDocuments?: IPassengerWithDocumentsAndBonusCardsDTO[];
    seats: number[];
    countries: ITrainCountrySelect[];
    initialValues?: IPassenger;
    tariffCategories: ITrainsTariffCategory[] | undefined;
    allowedDocumentTypes: TRAINS_DOCUMENT_TYPES[];
    isCppk: boolean;
    lastSegmentDeparture: string | undefined;
    onChangeError: (fieldName: string, meta: FieldMetaState<string>) => void;
}

class PassengerForm extends PureComponent<IPassengerFormProps> {
    /* Helpers */

    private getPassengerTitle(): string | null {
        const {ageGroup, tariffCategories} = this.props;

        return getPassengerTypeTitle(ageGroup, tariffCategories);
    }

    private validateBirthdate = (
        birthDate: string | undefined,
        formValues: ITrainsBookFormValues,
    ): string | undefined => {
        const {
            ageGroup,
            ageGroupIndex,
            lastSegmentDeparture,
            tariffCategories,
        } = this.props;

        // Проверка, что возраст пассажира соответствует тарифу
        const errorTariff =
            birthDate && lastSegmentDeparture && tariffCategories
                ? validateBirthdate(
                      birthDate,
                      ageGroup,
                      lastSegmentDeparture,
                      tariffCategories,
                  )
                : undefined;

        if (errorTariff) {
            return errorTariff;
        }

        if (ageGroup !== PASSENGERS_TYPES.ADULTS) {
            return;
        }

        // Проверка, что ребенка без места сопровождает взрослый старше десяти лет
        const {passengers, trainInfo} = formValues;
        const departureDate = trainInfo?.[EFieldName.startDate];

        if (!passengers || !departureDate) {
            return;
        }

        const {adults, babies} = passengers;

        if (!babies || !babies.length) {
            return;
        }

        const departureMoment = moment(departureDate, HUMAN_DATE_RU);

        const currentPassenger =
            formValues.passengers?.[ageGroup]?.[ageGroupIndex];
        const currentPassengerAge = currentPassenger?.birthdate
            ? getPassengerAge(currentPassenger.birthdate, departureMoment)
            : 0;

        if (currentPassengerAge >= 10) {
            return;
        }

        if (
            currentPassenger?.documentType &&
            DOCUMENT_TYPES_WITH_AGE_LIMIT.includes(
                currentPassenger?.documentType,
            )
        ) {
            return;
        }

        const adultsAges: {age: number; index: number}[] = adults
            ? adults.map((adult, index) => {
                  const adultBirthdate = adult.birthdate;

                  if (!adultBirthdate) {
                      return {age: 0, index};
                  }

                  return {
                      age: getPassengerAge(adultBirthdate, departureMoment),
                      index,
                  };
              })
            : [];

        const countAdultsOverTenYearsOld = adultsAges.filter(
            ({age}) => age >= 10,
        ).length;

        if (countAdultsOverTenYearsOld >= babies.length) {
            return;
        }

        // Ищем первого пассажира младше 10 лет
        const firstAdultsLtTenYearsOld = adultsAges.find(({age}) => age < 10);

        if (firstAdultsLtTenYearsOld?.index === ageGroupIndex) {
            return babiesDashWithDashAdultDashOverDashTenDashYearsDashOld();
        }

        return undefined;
    };

    private validateDocumentNumber = (
        documentNumberValue: string | undefined,
        formValues: ITrainsBookFormValues,
    ): string | undefined => {
        const {ageGroup, ageGroupIndex, isCppk} = this.props;
        const passengerValues =
            formValues.passengers?.[ageGroup]?.[ageGroupIndex];

        return validateDocumentNumber({
            documentNumberValue,
            formValues,
            passengerValues,
            ageGroup,
            isCppk,
        });
    };

    /* Render */

    private renderSectionTitle(): ReactNode {
        const {seats, passengerIndex} = this.props;
        const withPlaces = seats && seats.length > 0;

        return (
            <span>
                {`${passengerIndex + 1}. ${this.getPassengerTitle()}`}
                {withPlaces && (
                    <span
                        className={cx('seatNumber')}
                        {...prepareQaAttributes({
                            parent: this.props,
                            current: 'seats',
                        })}
                    >
                        {tariffKeyset.seats({
                            seats: seats.join(', '),
                            count: seats.length,
                        })}
                    </span>
                )}
            </span>
        );
    }

    render(): ReactNode {
        const {
            ageGroup,
            deviceType,
            countries,
            passengersWithDocuments,
            allowedDocumentTypes,
            initialValues,
            onChangeError,
        } = this.props;

        const availableDocuments = getDocumentList(
            ageGroup,
            allowedDocumentTypes,
        ).map(docType => MAP_TRAINS_DOCUMENT_TYPES_TO_COMMON[docType.value]);

        return (
            <div {...prepareQaAttributes(this.props)}>
                <BookingPassengerForm<ITrainsBookFormValues>
                    deviceType={deviceType}
                    title={this.renderSectionTitle()}
                    countries={countries}
                    passengers={passengersWithDocuments}
                    availableDocuments={availableDocuments}
                    validationInfo={validationInfo}
                    fieldsValidation={{
                        birthdate: this.validateBirthdate,
                        documentNumber: this.validateDocumentNumber,
                    }}
                    initialValues={initialValues}
                    excludeRuCitizenForOtherDocument
                    hideDisabledCitizenship
                    hasPatronymic
                    onChangeError={onChangeError}
                    {...prepareQaAttributes({
                        parent: this.props,
                        current: 'passengerForm',
                    })}
                />

                <AdditionFieldsFooter
                    className={cx(
                        deviceModDesktop('passengerFooter', deviceType),
                    )}
                    deviceType={deviceType}
                    ageGroup={ageGroup}
                    initialValues={initialValues}
                    {...prepareQaAttributes({
                        parent: this.props,
                        current: 'additionalFieldsFooter',
                    })}
                />

                <PassengerDataAdapter ageGroup={ageGroup} />
            </div>
        );
    }
}

export default PassengerForm;
