import {
    ReactNode,
    FunctionComponent,
    useMemo,
    useEffect,
    useContext,
} from 'react';
import {useForm} from 'react-final-form';
import {useSelector} from 'react-redux';

import {EDocumentType} from 'constants/document/documentTypes';
import {EFieldName as ECommonFieldName} from 'components/BookingPassengerForm/constants/fieldNames';
import {PASSENGERS_TYPES} from 'projects/trains/constants/passengersTypes';

import {IWithClassName} from 'types/withClassName';
import {IWithDeviceType} from 'types/withDeviceType';
import {
    IPassenger,
    EFieldName,
    ITrainsBookFormValues,
} from 'types/trains/booking/ITrainsBookFormValues';

import availableLoyaltyCardsSelector from 'selectors/trains/order/availableLoyaltyCardsSelector';
import trainListSelector from 'selectors/trains/order/trainListSelector';
import lastSegmentDepartureSelector from 'selectors/trains/order/passengers/lastSegmentDepartureSelector';

import {isValidDate} from 'utilities/dateUtils';
import {HUMAN_DATE_RU} from 'utilities/dateUtils/formats';
import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';
import {getPassengerTariff} from './utilities/getPassengerTariff';
import {getBoardingAge} from 'utilities/dateUtils/getBoardingAge';
import {getFilteredLoyaltyCards} from 'projects/trains/lib/order/fields/bonus';

import * as i18nBlock from 'i18n/trains-order-discount-documents';

import Form, {useFieldValue} from 'components/Form/Form';
import Checkbox from 'components/Checkbox/Checkbox';
import Separator from 'components/Separator/Separator';
import BonusCards from './components/BonusCards/BonusCards';

import cx from './AdditionFieldsFooter.scss';

interface IAdditionFieldsFooterProps
    extends IWithClassName,
        IWithDeviceType,
        IWithQaAttributes {
    ageGroup: PASSENGERS_TYPES;
    initialValues?: IPassenger;
}

const AdditionFieldsFooter: FunctionComponent<IAdditionFieldsFooterProps> = ({
    className,
    deviceType,
    ageGroup,
    initialValues,
    ...rest
}) => {
    const {isMobile} = deviceType;

    const loyaltyCards = useSelector(availableLoyaltyCardsSelector);
    const trains = useSelector(trainListSelector);
    const lastSegmentDeparture = useSelector(lastSegmentDepartureSelector);

    const formApi = useForm<ITrainsBookFormValues>();
    const groupId = useContext(Form.FieldGroupContext);
    const birthdate = useFieldValue<string>(ECommonFieldName.birthdate);
    const documentType = useFieldValue<EDocumentType>(
        ECommonFieldName.documentType,
    );

    const tariffNeedDocument = useMemo(() => {
        if (
            !lastSegmentDeparture ||
            !birthdate ||
            !isValidDate(birthdate || '', HUMAN_DATE_RU)
        ) {
            return false;
        }

        return trains.some(train => {
            if (!train.coach || !lastSegmentDeparture) {
                return false;
            }

            const passengerAge = getBoardingAge(
                birthdate,
                lastSegmentDeparture,
            );
            const tariff = getPassengerTariff(
                ageGroup,
                passengerAge,
                train.coach.tariffsInfo,
            );

            return tariff.needDocument;
        });
    }, [ageGroup, birthdate, trains, lastSegmentDeparture]);

    const availableBonusCardTypes = useMemo(() => {
        return documentType
            ? getFilteredLoyaltyCards(loyaltyCards, documentType)
            : [];
    }, [loyaltyCards, documentType]);

    /* Сброс значения при смене тарифа у пользователя */
    useEffect(() => {
        if (!tariffNeedDocument) {
            formApi.change(
                `${groupId}.${EFieldName.hasDiscountDocument}` as any,
                false,
            );
        }
    }, [tariffNeedDocument, formApi, groupId]);

    const isNotEmptyFooter =
        availableBonusCardTypes.length > 0 || tariffNeedDocument;

    return isNotEmptyFooter ? (
        <footer className={className} {...prepareQaAttributes(rest)}>
            {isMobile && <Separator margin="4" />}

            {availableBonusCardTypes.length > 0 && (
                <BonusCards
                    deviceType={deviceType}
                    bonusCardTypes={availableBonusCardTypes}
                    initialValues={initialValues}
                    {...prepareQaAttributes({
                        parent: rest,
                        current: 'loyaltyCards',
                    })}
                />
            )}

            {tariffNeedDocument && (
                <Form.Field
                    name={EFieldName.hasDiscountDocument}
                    initialValue={initialValues?.hasDiscountDocument}
                    render={({input}): ReactNode => (
                        <Checkbox
                            className={cx('discountDocumentСheckbox')}
                            size="m"
                            label={i18nBlock.hasDashDocumentDashPupil()}
                            checked={input.value}
                            onChange={input.onChange}
                            onFocus={input.onFocus}
                            onBlur={input.onBlur}
                        />
                    )}
                />
            )}
        </footer>
    ) : null;
};

export default AdditionFieldsFooter;
