import {FunctionComponent, useCallback, useContext} from 'react';
import {useForm} from 'react-final-form';

import {TRAIN_BONUS_CARDS} from 'projects/trains/constants/bonusCards';

import {IWithDeviceType} from 'types/withDeviceType';
import {IFieldError} from 'types/common/validation/form';
import {
    EFieldName,
    IPassenger,
    ITrainsBookFormValues,
} from 'types/trains/booking/ITrainsBookFormValues';

import {deviceMods} from 'utilities/stylesUtils';
import {useBoolean} from 'utilities/hooks/useBoolean';
import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';
import {haveBonusCardError} from './utilities/haveBonusCardError';

import * as i18nBlock from 'i18n/trains-loyalty-cards';

import Form from 'components/Form/Form';
import LinkButton from 'components/LinkButton/LinkButton';
import RotatingArrowIcon from 'components/RotatingArrowIcon/RotatingArrowIcon';
import AutoHeightAnimateControl from 'components/AutoHeightAnimateControl/AutoHeightAnimateControl';
import BonusCard from './components/BonusCard/BonusCard';

import cx from './BonusCards.scss';

interface IBonusCardsProps extends IWithQaAttributes, IWithDeviceType {
    bonusCardTypes: TRAIN_BONUS_CARDS[];
    initialValues?: IPassenger;
}

const BonusCards: FunctionComponent<IBonusCardsProps> = ({
    deviceType,
    bonusCardTypes,
    initialValues,
    ...rest
}) => {
    const cardsLength = bonusCardTypes.length;

    const formApi = useForm<ITrainsBookFormValues>();
    const formGroupId = useContext(Form.FieldGroupContext);

    const {
        value: isExpanded,
        toggle: toggleExpand,
        setTrue: expand,
    } = useBoolean(Boolean(initialValues?.isExpandedBonusCard));

    const handleSubmit = useCallback(
        (
            errors?: string | IFieldError,
            submitErrors?: string | IFieldError,
        ) => {
            if (isExpanded) {
                return;
            }

            const haveBonusCardErrors = haveBonusCardError(
                errors,
                submitErrors,
            );

            if (haveBonusCardErrors) {
                expand();
            }
        },
        [isExpanded, expand],
    );

    /* Показываем поля ввода, после чего раскрываем их */
    const handleExpandLinkClick = useCallback(() => {
        formApi.change(
            `${formGroupId}.${EFieldName.isExpandedBonusCard}` as any,
            !isExpanded,
        );
        toggleExpand();
    }, [toggleExpand, isExpanded, formApi, formGroupId]);

    return (
        <div
            className={cx(deviceMods('root', deviceType))}
            {...prepareQaAttributes(rest)}
        >
            <LinkButton
                className={cx('toggleLink')}
                onClick={handleExpandLinkClick}
                {...prepareQaAttributes({parent: rest, current: 'toggleLink'})}
            >
                {cardsLength > 1
                    ? i18nBlock.subtitleDashFull()
                    : i18nBlock.collapseDashTitleDashRzhdb()}
                <RotatingArrowIcon
                    className={cx('toggleLinkIcon')}
                    rotated={isExpanded}
                />
            </LinkButton>
            <Form.FieldGroup groupId={EFieldName.bonusCards}>
                <AutoHeightAnimateControl canShowComponent={isExpanded}>
                    <div className={cx('cards')}>
                        {bonusCardTypes.map((cardType, index) => (
                            <BonusCard
                                key={cardType}
                                className={cx(
                                    'cardInputControl',
                                    `cardInputControl_index_${index}`,
                                )}
                                deviceType={deviceType}
                                bonusCardType={cardType}
                                isExpanded={isExpanded}
                                isDefaultTitle={cardsLength === 1}
                                initialValue={
                                    initialValues?.bonusCards?.[cardType]
                                }
                                onSubmit={handleSubmit}
                                {...prepareQaAttributes({
                                    parent: rest,
                                    current: cardType,
                                })}
                            />
                        ))}
                    </div>
                </AutoHeightAnimateControl>
            </Form.FieldGroup>
        </div>
    );
};

export default BonusCards;
