import {
    ReactNode,
    FunctionComponent,
    useMemo,
    useEffect,
    useContext,
} from 'react';
import {useField} from 'react-final-form';

import {TRAIN_BONUS_CARDS} from 'projects/trains/constants/bonusCards';

import {IWithClassName} from 'types/withClassName';
import {IWithDeviceType} from 'types/withDeviceType';
import {IFieldError} from 'types/common/validation/form';

import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';
import {BONUS_CARD_MASK} from 'projects/trains/lib/order/fields/masks';
import {validateBonusCardDuplication} from './utilities/validateBonusCard';
import getLoyaltyCardName from 'projects/trains/lib/getLoyaltyCardName';

import * as i18nBlock from 'i18n/trains-loyalty-cards';

import Form from 'components/Form/Form';
import MaskedInput from 'components/MaskedInput/MaskedInput';
import FormField from 'components/FormField/FormField';

interface IBonusCardProps
    extends IWithClassName,
        IWithDeviceType,
        IWithQaAttributes {
    bonusCardType: TRAIN_BONUS_CARDS;
    isDefaultTitle: boolean;
    isExpanded: boolean;
    initialValue?: string;
    onSubmit: (
        errors?: string | IFieldError,
        submitErrors?: string | IFieldError,
    ) => void;
}

const BonusCard: FunctionComponent<IBonusCardProps> = ({
    className,
    deviceType,
    bonusCardType,
    isDefaultTitle,
    isExpanded,
    initialValue,
    onSubmit,
    ...rest
}) => {
    const formGroupId = useContext(Form.FieldGroupContext);
    const field = useField(`${formGroupId}.${bonusCardType}`, {
        subscription: {data: true, error: true, submitError: true},
    });
    const submitAmount = field.meta.data?.submitAmount;

    useEffect(() => {
        if (submitAmount) {
            onSubmit(field.meta.error, field.meta.submitError);
        }
    }, [submitAmount]);

    const title = useMemo(() => {
        return isDefaultTitle
            ? i18nBlock.fieldTitle()
            : getLoyaltyCardName(bonusCardType);
    }, [isDefaultTitle, bonusCardType]);

    return (
        <FormField
            className={className}
            name={bonusCardType}
            title={title}
            deviceType={deviceType}
            initialValue={initialValue}
            validate={validateBonusCardDuplication}
            control={({input}, {error, inputRef, controlRef}): ReactNode => (
                <MaskedInput
                    {...input}
                    id={input.name}
                    inputRef={(currentInputRef): void => {
                        inputRef.current = currentInputRef;
                    }}
                    inputMode="numeric"
                    innerRef={controlRef}
                    value={input.value || null}
                    size="l"
                    state={error ? 'error' : undefined}
                    type="text"
                    tabIndex={isExpanded ? undefined : -1}
                    mask={BONUS_CARD_MASK}
                    {...prepareQaAttributes(rest)}
                />
            )}
        />
    );
};

export default BonusCard;
