import {EDocumentType} from 'constants/document/documentTypes';
import {PASSENGERS_TYPES} from 'projects/trains/constants/passengersTypes';

import {isNotNullable} from 'types/utilities';
import {
    ITrainsBookFormValues,
    IPassenger,
} from 'types/trains/booking/ITrainsBookFormValues';

import {flatMapPassengers} from 'projects/trains/lib/order/passengers/flatPassengers';

import * as i18nBlock from 'i18n/trains-order-fields-errors';

interface IValidationData {
    documentNumberValue: string | undefined;
    passengerValues?: IPassenger;
    formValues: ITrainsBookFormValues;
    ageGroup: PASSENGERS_TYPES;
    isCppk: boolean;
}

type TValidator = (validationData: IValidationData) => string | undefined;

/** Проверяет на одинаковые номера документов в поездах ЦППК */
const checkEqualDocumentNumbers: TValidator = ({
    documentNumberValue,
    formValues,
    isCppk,
}) => {
    if (!isCppk || !formValues.passengers || !documentNumberValue) {
        return undefined;
    }

    const documentNumbers = flatMapPassengers(
        formValues.passengers,
        passenger => passenger.documentNumber,
    ).filter(isNotNullable);

    return documentNumbers.filter(
        documentNumber => documentNumber === documentNumberValue,
    ).length > 1
        ? i18nBlock.documentNumberDotDuplicate()
        : undefined;
};

function isSameName(
    currentPassenger: IPassenger | undefined,
    passenger: IPassenger,
): boolean {
    return Boolean(
        currentPassenger?.firstName &&
            currentPassenger?.lastName &&
            passenger.firstName?.trim() ===
                currentPassenger?.firstName?.trim() &&
            passenger.lastName?.trim() === currentPassenger?.lastName?.trim() &&
            passenger.patronymicName?.trim() ===
                currentPassenger?.patronymicName?.trim(),
    );
}

/** Проверяет на одинаковые номера документов детей */
const checkEqualChildrenDocumentNumbers: TValidator = ({
    documentNumberValue,
    formValues,
    passengerValues,
    ageGroup,
}) => {
    if (
        ageGroup === PASSENGERS_TYPES.ADULTS ||
        !formValues.passengers ||
        !documentNumberValue
    ) {
        return undefined;
    }

    const {children, babies} = formValues.passengers;
    const sameDocumentNumberPassengers = flatMapPassengers(
        {children, babies},
        passenger =>
            passenger.documentNumber === documentNumberValue ? passenger : null,
    ).filter(isNotNullable);

    if (sameDocumentNumberPassengers.length <= 1) {
        return undefined;
    }

    if (passengerValues?.documentType === EDocumentType.RU_BIRTH_CERTIFICATE) {
        return i18nBlock.documentNumberDotDuplicateBirthCertificate();
    }

    const fullySamePassengers = sameDocumentNumberPassengers.filter(passenger =>
        isSameName(passengerValues, passenger),
    );

    return fullySamePassengers.length > 1
        ? i18nBlock.documentNumberDotChildrenDuplicate()
        : undefined;
};

export function validateDocumentNumber(
    validationData: IValidationData,
): string | undefined {
    return (
        checkEqualDocumentNumbers(validationData) ||
        checkEqualChildrenDocumentNumbers(validationData)
    );
}
