import {useCallback, useEffect, useMemo, useRef} from 'react';
import {useDispatch, useSelector} from 'react-redux';
import {batchActions} from 'redux-batched-actions';
import {AnyAction} from 'redux';

import {TRAIN_COACH_TYPE} from 'projects/trains/constants/coachType';
import {SAPSAN_TRAIN_NAME} from 'projects/trains/constants/rawTrainNames';

import {ETrainsGoal} from 'utilities/metrika/types/goals/trains';

import {
    setExpandedServiceClass,
    setOrderCoachType,
} from 'reducers/trains/order/actions/trains';

import currentTrainDetailsCoachesWithEnoughPlacesSelector from 'selectors/trains/order/currentTrainDetailsCoachesWithEnoughPlacesSelector';
import currentTrainDetailsSelector from 'selectors/trains/order/currentTrainDetailsSelector';
import currentSegmentDirectionAndIndexSelector from 'selectors/trains/order/currentSegmentDirectionAndIndexSelector';

import {reachGoal} from 'utilities/metrika';
import getCoachesGroupedByTypeWithInfo from 'projects/trains/lib/order/coaches/getCoachesGroupedByTypeWithInfo';
import getClassesCount from 'projects/trains/lib/order/coaches/getClassesCount';
import {prepareQaAttributes} from 'utilities/qaAttributes/qaAttributes';
import getGoalByDirectionAndIndex from 'projects/trains/components/TrainsOrderPage/PlacesStep/utilities/getGoalByDirectionAndIndex';

import BookingLayout from 'components/Layouts/BookingLayout/BookingLayout';
import PassengersCountSection from 'projects/trains/components/TrainsOrderPage/PassengersCountSection/PassengersCountSection';
import UpdateNotification from 'projects/trains/components/UpdateNotification/UpdateNotification';
import SimpleSelectorView from 'projects/trains/components/TrainsOrderPage/SimpleSelectorView/SimpleSelectorView';

import {PlacesStepScopeRefContext} from './contexts/PlacesStepScopeRefContext';

const PLACES_STEP_QA = 'placesStep';

const TrainsOrderPlacesStep: React.FC = () => {
    const dispatch = useDispatch();

    const coachesWithPlacesForPassengers = useSelector(
        currentTrainDetailsCoachesWithEnoughPlacesSelector,
    );
    const trainDetails = useSelector(currentTrainDetailsSelector);
    const currentDirectionAndIndex = useSelector(
        currentSegmentDirectionAndIndexSelector,
    );
    const placesStepScopeRef = useRef<HTMLElement>(null);

    useEffect(() => reachGoal(ETrainsGoal.ORDER_PLACES_PAGE_LOAD), []);
    useEffect(() => {
        const goal = getGoalByDirectionAndIndex(currentDirectionAndIndex);

        if (goal) {
            reachGoal(goal);
        }
    }, [currentDirectionAndIndex]);

    const handleNotificationUpdate = useCallback(() => {
        reachGoal(ETrainsGoal.PLACES_EXPIRED_REFRESH_CLICK);
    }, []);

    const handleUpdateNotificationSkip = useCallback(() => {
        reachGoal(ETrainsGoal.PLACES_EXPIRED_CONTINUE);
    }, []);

    const onServiceClassChange = useCallback(
        (serviceClassKey: string, coachType: TRAIN_COACH_TYPE | null) => {
            const actions: AnyAction[] = [
                setExpandedServiceClass({
                    ...currentDirectionAndIndex,
                    data: serviceClassKey,
                }),
            ];

            if (coachType) {
                actions.push(
                    setOrderCoachType({
                        ...currentDirectionAndIndex,
                        data: coachType,
                    }),
                );
            }

            dispatch(batchActions(actions));
        },
        [dispatch, currentDirectionAndIndex],
    );

    const handleShowUpdateNotification = useCallback(() => {
        reachGoal(ETrainsGoal.PLACES_EXPIRED);
    }, []);

    const coachGroupsWithClasses = useMemo(() => {
        if (!trainDetails) {
            return null;
        }

        const {schemas} = trainDetails;

        return getCoachesGroupedByTypeWithInfo({
            schemas,
            beddingOption: true,
            coaches: coachesWithPlacesForPassengers,
        });
    }, [trainDetails, coachesWithPlacesForPassengers]);

    useEffect(() => {
        if (!coachGroupsWithClasses) {
            return;
        }

        if (getClassesCount(coachGroupsWithClasses) === 1) {
            const singleClass = coachGroupsWithClasses[0];

            onServiceClassChange(singleClass.classes[0].key, singleClass.type);
        }
    }, [coachGroupsWithClasses, onServiceClassChange]);

    if (!coachGroupsWithClasses) {
        return null;
    }

    return (
        <PlacesStepScopeRefContext.Provider value={placesStepScopeRef}>
            <BookingLayout.Forms
                ref={placesStepScopeRef}
                {...prepareQaAttributes(PLACES_STEP_QA)}
            >
                <BookingLayout.Card>
                    <PassengersCountSection
                        {...prepareQaAttributes('passengersCountSection')}
                    />
                </BookingLayout.Card>
                <SimpleSelectorView
                    trainDetails={trainDetails}
                    allCoaches={trainDetails?.coaches ?? []}
                    coachesWithEnoughPlaces={coachesWithPlacesForPassengers}
                    forceClassSelector={
                        trainDetails?.rawTrainName === SAPSAN_TRAIN_NAME
                    }
                    {...prepareQaAttributes('coachTabsSection')}
                />
                <UpdateNotification
                    onUpdate={handleNotificationUpdate}
                    onShowNotification={handleShowUpdateNotification}
                    onSkip={handleUpdateNotificationSkip}
                />
            </BookingLayout.Forms>
        </PlacesStepScopeRefContext.Provider>
    );
};

export default TrainsOrderPlacesStep;
