import React from 'react';

import {IWithClassName} from 'types/withClassName';
import {isNotUndefined} from 'types/utilities';

import IPrice from 'utilities/currency/PriceInterface';
import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';

import * as i18nBlock from 'i18n/trains-order-fee';

import InfoPopup from 'components/InfoPopup/InfoPopup';
import PriceItem from './components/PriceItem/PriceItem';

import cx from './PriceExplanation.scss';

export interface IPriceExplanationProps
    extends IWithClassName,
        IWithQaAttributes {
    ticketsPrice: IPrice | undefined;
    feePrice: IPrice | undefined;
    insurancePrice?: IPrice | undefined;
    isMultipleTickets?: boolean;
    iconClassName?: string;
    scopeRef?: React.RefObject<HTMLElement>;
}

const PriceExplanation: React.FC<IPriceExplanationProps> = props => {
    const {
        className,
        ticketsPrice,
        feePrice,
        insurancePrice,
        isMultipleTickets = false,
        iconClassName,
        scopeRef,
    } = props;

    return (
        <InfoPopup
            className={className}
            iconClassName={iconClassName}
            scopeRef={scopeRef}
            {...prepareQaAttributes({parent: props, current: 'questionIcon'})}
        >
            <div {...prepareQaAttributes(props)}>
                {[
                    {
                        title: isMultipleTickets
                            ? i18nBlock.fullDashAmountDashAll()
                            : i18nBlock.fullDashAmount(),
                        price: ticketsPrice,
                        key: 'ticketsPrice',
                    },
                    {
                        title: i18nBlock.fee(),
                        price: feePrice,
                        key: 'feePrice',
                    },
                    {
                        title: i18nBlock.insurance(),
                        price: insurancePrice,
                        key: 'insurancePrice',
                    },
                ]
                    .filter(({price}) => isNotUndefined(price) && price.value)
                    .map(({title, price, key}) => (
                        <PriceItem
                            key={key}
                            className={cx('row')}
                            {...prepareQaAttributes({
                                key,
                                parent: props,
                                current: 'priceItem',
                            })}
                            title={title}
                            price={price}
                        />
                    ))}
            </div>
        </InfoPopup>
    );
};

export default PriceExplanation;
