import React, {useMemo, useCallback} from 'react';
import range from 'lodash/range';

import {MAX_PLACE_TYPE_COUNT} from 'projects/trains/constants/requirements';
import {SAPSAN_TRAIN_NAME} from 'projects/trains/constants/rawTrainNames';
import {TRAIN_COACH_TYPE} from 'projects/trains/constants/coachType';

import {
    CountRequirementsType,
    ITrainsCoach,
    ITrainsRequirements,
    TrainsPassengersCount,
} from 'reducers/trains/order/types';
import {ITrainsSchema} from 'server/api/TrainsApi/types/ITrainsDetailsApiResponse';
import {IWithClassName} from 'types/withClassName';
import {isNotNull} from 'types/utilities';

import {
    clearCountRequirements,
    setCountRequirements,
} from 'reducers/trains/order/actions/trains';

import isArrangementCountOptionAvailable from 'projects/trains/components/TrainsOrderPage/Requirements/ArrangementCountRequirement/helpers/isArrangementCountOptionAvailable';
import {isNotEmptyCountField} from 'projects/trains/lib/order/utils';
import areAllCoachesWithSchemas from 'projects/trains/lib/order/coaches/areAllCoachesWithSchemas';
import {countPassengersWithPlaces} from 'projects/trains/lib/order/passengers/utils';
import getTextForPlaceType from 'projects/trains/components/TrainsOrderPage/Requirements/ArrangementCountRequirement/helpers/getTextForPlaceType';
import {useDeviceType} from 'utilities/hooks/useDeviceType';

import * as i18nTrainsTransportSchemaBlock from 'i18n/trains-transport-schema';
import * as i18nBlock from 'i18n/trains-place-categories';

import Select from 'components/Select/Select';

import cx from './ArrangementCountRequirement.scss';

interface IArrangementCountRequirementProps extends IWithClassName {
    coachType: TRAIN_COACH_TYPE;
    trainName: string;
    requirements: ITrainsRequirements;
    coaches: ITrainsCoach[];
    schemas: Record<number, ITrainsSchema>;
    passengers: TrainsPassengersCount;
    onClearCountRequirements(
        actionCreator: typeof clearCountRequirements,
        data: undefined,
    ): void;
    onSetCountRequirements(
        actionCreator: typeof setCountRequirements,
        data: CountRequirementsType,
    ): void;
}

const ArrangementCountRequirement: React.FC<IArrangementCountRequirementProps> =
    props => {
        const {
            className,
            coachType,
            trainName,
            schemas,
            requirements,
            requirements: {arrangement},
            coaches,
            passengers,
            onClearCountRequirements,
            onSetCountRequirements,
        } = props;
        const {isMobile} = useDeviceType();

        const selectedValue = useMemo((): string | null => {
            if (isNotEmptyCountField(requirements)) {
                const countRequirements = requirements.count;

                if (coachType === TRAIN_COACH_TYPE.SITTING) {
                    const nearWindow =
                        (countRequirements &&
                            'nearWindow' in countRequirements &&
                            countRequirements.nearWindow) ||
                        0;
                    const nearPassage =
                        (countRequirements &&
                            'nearPassage' in countRequirements &&
                            countRequirements.nearPassage) ||
                        0;

                    return [nearWindow, nearPassage].join(',');
                }

                const upper =
                    (countRequirements &&
                        'upper' in countRequirements &&
                        countRequirements.upper) ||
                    0;
                const bottom =
                    (countRequirements &&
                        'bottom' in countRequirements &&
                        countRequirements.bottom) ||
                    0;

                return [upper, bottom].join(',');
            }

            return null;
        }, [requirements, coachType]);

        const setsOfSeats = useMemo((): {
            value: string | null;
            data: string;
        }[] => {
            const typeCounts = MAX_PLACE_TYPE_COUNT[coachType];

            const passengersCount = countPassengersWithPlaces(passengers);
            const maxArrangementCombinationsCount =
                (arrangement && typeCounts && typeCounts[arrangement]) || null;
            const allCoachesAreWithSchemas = areAllCoachesWithSchemas(coaches);

            const options = range(passengersCount + 1)
                .map((set, i) => {
                    const value: [number, number] = [i, passengersCount - i];

                    if (
                        maxArrangementCombinationsCount &&
                        Math.max(...value) > maxArrangementCombinationsCount
                    ) {
                        return null;
                    }

                    if (
                        allCoachesAreWithSchemas &&
                        !isArrangementCountOptionAvailable({
                            option: value,
                            coaches,
                            schemas,
                            passengersCount,
                            coachType,
                            requirements,
                        })
                    ) {
                        return null;
                    }

                    return {
                        value: value.join(','),
                        data: getTextForPlaceType(coachType, value),
                    };
                })
                .filter(isNotNull);

            return [
                {
                    value: null,
                    data: i18nBlock.requirementDashIrrelevant(),
                },
                ...options,
            ];
        }, [
            arrangement,
            coachType,
            coaches,
            passengers,
            requirements,
            schemas,
        ]);

        const handleSelectChange = useCallback(
            (value: string | null) => {
                if (value === null) {
                    onClearCountRequirements(clearCountRequirements, undefined);

                    return;
                }

                const [firstParam, secondParam] = value
                    .split(',')
                    .map(v => Number.parseInt(v, 10));

                if (coachType === TRAIN_COACH_TYPE.SITTING) {
                    onSetCountRequirements(setCountRequirements, {
                        nearWindow: firstParam,
                        nearPassage: secondParam,
                    });
                } else {
                    onSetCountRequirements(setCountRequirements, {
                        upper: firstParam,
                        bottom: secondParam,
                    });
                }
            },
            [coachType, onClearCountRequirements, onSetCountRequirements],
        );

        if (trainName === SAPSAN_TRAIN_NAME) {
            return null;
        }

        return (
            <div className={cx('root', className)}>
                <div className={cx('title')}>
                    {i18nTrainsTransportSchemaBlock.requirementsDashTitleDashPlaceDashType()}
                </div>
                <Select
                    theme="outlined"
                    placeholder={i18nTrainsTransportSchemaBlock.requirementsDashPlaceDashTypeDashPlaceholder()}
                    size={isMobile ? 'l' : 'm'}
                    onChange={handleSelectChange}
                    options={setsOfSeats}
                    value={selectedValue}
                    width="max"
                />
            </div>
        );
    };

export default ArrangementCountRequirement;
