import React, {useCallback, useRef} from 'react';
import {useDispatch} from 'react-redux';

import {IWithClassName} from 'types/withClassName';
import {ITrainsCoach, ITrainsDetails} from 'reducers/trains/order/types';
import {TTrainsStoreOrderSegment} from 'projects/trains/lib/segments/types';
import EPopupDirection from 'components/Popup/types/EPopupDirection';

import {changeCoachAndResetRequirements} from 'reducers/trains/actions/changeCoachAndResetRequirements';

import needToShowTrainsCoachNumberRequirement from 'projects/trains/components/TrainsOrderPage/Requirements/helpers/needToShowTrainsCoachNumberRequirement';
import isTrainsDifferentCoachesArrival from 'projects/trains/lib/order/coaches/isTrainsDifferentCoachesArrival';
import {useDeviceType} from 'utilities/hooks/useDeviceType';

import * as i18nBlock from 'i18n/trains-train-details';

import CoachHeader from 'projects/trains/components/TrainsOrderPage/Coach/components/CoachHeader/CoachHeader';
import Dropdown, {
    IDropdownPopupParams,
    IDropdownSwitcherParams,
} from 'components/Dropdown/Dropdown';
import Button from 'components/Button/Button';
import CoachSelectorItem from 'projects/trains/components/TrainsOrderPage/Requirements/CoachNumber/CoachSelectorItem/CoachSelectorItem';
import ArrowBottomIcon from 'icons/12/ArrowBottom';
import ArrowTopIcon from 'icons/12/ArrowTop';

import cx from './CoachNumber.scss';

const POPUP_PROPS = {
    directions: [EPopupDirection.BOTTOM_LEFT],
    hasTail: true,
    plain: true,
    nonvisual: false,
};

interface ICoachNumberRequirementProps extends IWithClassName {
    segment: TTrainsStoreOrderSegment | null;
    coach: ITrainsCoach | null;
    trainDetails: ITrainsDetails | null | undefined;
    coaches: ITrainsCoach[];
}

interface ICoachNumberButtonProps extends ICoachNumberRequirementProps {}

const CoachNumberButton: React.FC<
    IDropdownSwitcherParams<ICoachNumberButtonProps>
> = ({meta, input, componentProps}) => {
    const {isMobile} = useDeviceType();
    const {visible, hidePopup, showPopup} = meta;
    const {segment, coach, trainDetails, coaches} = componentProps;

    const clickHandler = visible ? hidePopup : showPopup;
    const activeCoach = coach && coaches.includes(coach) ? coach : coaches[0];

    if (!segment) {
        return null;
    }

    return (
        <div className={cx('buttonWrapper')} ref={input.ref}>
            <Button
                className={cx('dropdownButton')}
                size="m-inset"
                theme="plain"
                onClick={clickHandler}
                width={isMobile ? 'max' : 'auto'}
                iconRight={visible ? <ArrowTopIcon /> : <ArrowBottomIcon />}
            >
                <CoachHeader
                    className={cx('coachHeader')}
                    coach={activeCoach}
                    inline
                    withoutCoachNumberTitle={false}
                    showNFCoach={Boolean(
                        trainDetails?.isFirm && !activeCoach.isFirm,
                    )}
                    showCoachArrival={isTrainsDifferentCoachesArrival(coaches)}
                />
            </Button>
        </div>
    );
};

interface ICoachNumberPopupProps extends ICoachNumberRequirementProps {
    handleCoachChange: (coach: ITrainsCoach) => void;
}

const CoachNumberPopup: React.FC<IDropdownPopupParams<ICoachNumberPopupProps>> =
    ({componentProps}) => {
        const {
            segment,
            trainDetails = null,
            coaches,
            handleCoachChange,
        } = componentProps;

        return (
            <div className={cx('popup')}>
                {coaches.map((item, index) => (
                    <CoachSelectorItem
                        key={index}
                        coach={item}
                        segment={segment}
                        trainDetails={trainDetails}
                        showCoachArrival={isTrainsDifferentCoachesArrival(
                            coaches,
                        )}
                        onClick={handleCoachChange}
                    />
                ))}
            </div>
        );
    };

const CoachNumberRequirement: React.FC<ICoachNumberRequirementProps> = ({
    segment,
    coach,
    trainDetails,
    coaches,
    className,
}) => {
    const dispatch = useDispatch();
    const dropDownRef = useRef<Dropdown>();
    const handleCoachChange = useCallback(
        (coach: ITrainsCoach): void => {
            dispatch(changeCoachAndResetRequirements(coach));

            if (dropDownRef.current) {
                dropDownRef.current.hidePopup();
            }
        },
        [dispatch, dropDownRef],
    );

    if (!needToShowTrainsCoachNumberRequirement(coaches)) {
        return null;
    }

    return (
        <div className={cx('root', className)}>
            <div className={cx('selectorTitle')}>
                {i18nBlock.chooseDashCoachDashSelectorDashTitle()}
            </div>

            <Dropdown<ICoachNumberButtonProps, ICoachNumberPopupProps>
                // @ts-ignore
                ref={dropDownRef}
                switcherComponent={CoachNumberButton}
                switcherComponentProps={{
                    segment,
                    coach,
                    trainDetails,
                    coaches,
                }}
                popupComponent={CoachNumberPopup}
                popupProps={POPUP_PROPS}
                popupComponentProps={{
                    segment,
                    coach,
                    trainDetails,
                    coaches,
                    handleCoachChange,
                }}
            />
        </div>
    );
};

export default CoachNumberRequirement;
