import React, {useMemo, useCallback} from 'react';

import {ARRANGEMENT_REQUIREMENTS} from 'projects/trains/constants/requirements';
import {SAPSAN_TRAIN_NAME} from 'projects/trains/constants/rawTrainNames';
import {SITTING} from 'projects/trains/lib/segments/tariffs/constants';
import {TRAIN_COACH_TYPE} from 'projects/trains/constants/coachType';

import {
    ITrainsCoach,
    ITrainsRequirements,
    TrainsPassengersCount,
} from 'reducers/trains/order/types';
import {ITrainsSchema} from 'server/api/TrainsApi/types/ITrainsDetailsApiResponse';
import {IWithClassName} from 'types/withClassName';

import {setArrangementRequirements} from 'reducers/trains/order/actions/trains';

import isNearWindowAndPassageCountsOptionAvailable from 'projects/trains/components/TrainsOrderPage/Requirements/helpers/isNearWindowAndPassageCountsOptionAvailable';
import areAllCoachesWithSchemas from 'projects/trains/lib/order/coaches/areAllCoachesWithSchemas';
import {countPassengersWithPlaces} from 'projects/trains/lib/order/passengers/utils';

import * as i18nBlock from 'i18n/trains-place-categories';

import Checkbox from 'components/Checkbox/Checkbox';

interface INearestRequirementProps extends IWithClassName {
    value?: ARRANGEMENT_REQUIREMENTS;
    requirements: ITrainsRequirements;
    coachType: TRAIN_COACH_TYPE;
    passengers: TrainsPassengersCount;
    schemas: Record<number, ITrainsSchema>;
    coaches: ITrainsCoach[];
    trainName: string;
    onSetRequirements(
        actionCreator: typeof setArrangementRequirements,
        data: ARRANGEMENT_REQUIREMENTS,
    ): void;
}

const NearestRequirement: React.FC<INearestRequirementProps> = props => {
    const {
        className,
        value,
        requirements,
        coachType,
        passengers,
        schemas,
        coaches,
        trainName,
        onSetRequirements,
    } = props;

    const isCheckboxDisabled = useMemo(() => {
        if (!areAllCoachesWithSchemas(coaches)) {
            return false;
        }

        const placeCounts = requirements.count || null;
        const nearWindow =
            (placeCounts &&
                'nearWindow' in placeCounts &&
                placeCounts.nearWindow) ||
            0;
        const nearPassage =
            (placeCounts &&
                'nearPassage' in placeCounts &&
                placeCounts.nearPassage) ||
            0;

        const passengersCount = countPassengersWithPlaces(passengers);
        const oppositeCheckboxValue =
            value === ARRANGEMENT_REQUIREMENTS.NEAREST
                ? ARRANGEMENT_REQUIREMENTS.IRRELEVANT
                : ARRANGEMENT_REQUIREMENTS.NEAREST;

        // Если при противоположном значении чекбокса, вариант с текущим количеством мест (у окна, у прохода) доступен,
        // значит чекбокс доступен для переключения.
        return !isNearWindowAndPassageCountsOptionAvailable({
            option: [nearWindow, nearPassage],
            coaches,
            schemas,
            passengersCount,
            coachType,
            requirements: {
                arrangement: oppositeCheckboxValue,
            },
        });
    }, [coachType, requirements, coaches, passengers, schemas, value]);

    const onNearestPlaceChange = useCallback(
        e => {
            const {checked} = e.target;

            onSetRequirements(
                setArrangementRequirements,
                checked
                    ? ARRANGEMENT_REQUIREMENTS.NEAREST
                    : ARRANGEMENT_REQUIREMENTS.IRRELEVANT,
            );
        },
        [onSetRequirements],
    );

    const passengersCount = countPassengersWithPlaces(passengers);

    if (
        coachType !== SITTING ||
        passengersCount !== 2 ||
        trainName === SAPSAN_TRAIN_NAME
    ) {
        return null;
    }

    return (
        <Checkbox
            className={className}
            size="m"
            label={i18nBlock.sittingDashNearest()}
            checked={value === ARRANGEMENT_REQUIREMENTS.NEAREST}
            disabled={isCheckboxDisabled}
            onChange={onNearestPlaceChange}
        />
    );
};

export default NearestRequirement;
