import React, {useCallback, useMemo} from 'react';

import {ARRANGEMENT_REQUIREMENTS} from 'projects/trains/constants/requirements';
import {PLATZKARTE} from 'projects/trains/lib/segments/tariffs/constants';
import {TRAIN_COACH_TYPE} from 'projects/trains/constants/coachType';

import {IWithClassName} from 'types/withClassName';
import {
    ITrainsCoach,
    ITrainsRequirements,
    TrainsPassengersCount,
} from 'reducers/trains/order/types';
import {ITrainsSchema} from 'server/api/TrainsApi/types/ITrainsDetailsApiResponse';

import {setArrangementRequirements} from 'reducers/trains/order/actions/trains';

import isArrangementOptionAvailable from 'projects/trains/components/TrainsOrderPage/Requirements/helpers/isArrangementOptionAvailable';
import {countPassengersWithPlaces} from 'projects/trains/lib/order/passengers/utils';
import {useDeviceType} from 'utilities/hooks/useDeviceType';
import getArrangementRequirements from 'projects/trains/components/TrainsOrderPage/Requirements/PlatzkarteRequirement/utilities/getArrangementRequirements';

import Radiobox from 'components/Radiobox/Radiobox';

import cx from './PlatzkarteRequirement.scss';

interface IButtonOptons {
    value: ARRANGEMENT_REQUIREMENTS;
    label: string;
    disabled: boolean;
}

const buildOption = (
    value: ARRANGEMENT_REQUIREMENTS,
    disabled: boolean = false,
): IButtonOptons => ({
    value,
    label: getArrangementRequirements(value),
    disabled,
});

const getButtons = (
    passengersCount: number,
    requirements: ITrainsRequirements,
    coaches: ITrainsCoach[],
    schemas: Record<number, ITrainsSchema>,
    coachType: TRAIN_COACH_TYPE,
): IButtonOptons[] => {
    const optionsToCheck = [
        ARRANGEMENT_REQUIREMENTS.COMPARTMENT,
        ARRANGEMENT_REQUIREMENTS.SECTION,
        ARRANGEMENT_REQUIREMENTS.NOT_SIDE,
    ];

    const [compartmentIsDisabled, sectionIsDisabled, notSideIsDisabled] =
        optionsToCheck.map(
            option =>
                !isArrangementOptionAvailable({
                    option,
                    requirements,
                    coaches,
                    schemas,
                    passengersCount,
                    coachType,
                }),
        );

    return [
        buildOption(
            ARRANGEMENT_REQUIREMENTS.COMPARTMENT,
            compartmentIsDisabled,
        ),
        buildOption(ARRANGEMENT_REQUIREMENTS.SECTION, sectionIsDisabled),
        buildOption(ARRANGEMENT_REQUIREMENTS.NOT_SIDE, notSideIsDisabled),
        buildOption(ARRANGEMENT_REQUIREMENTS.IRRELEVANT),
    ];
};

function getButtonsKey(buttons: IButtonOptons[]): string {
    return buttons.map(({value, disabled}) => `${value}:${disabled}`).join(';');
}

interface IPlatzkarteRequirementProps extends IWithClassName {
    value?: ARRANGEMENT_REQUIREMENTS;
    coachType: TRAIN_COACH_TYPE;
    passengers: TrainsPassengersCount;
    requirements: ITrainsRequirements;
    coaches: ITrainsCoach[];
    schemas: Record<number, ITrainsSchema>;
    onSetRequirements(
        actionCreator: typeof setArrangementRequirements,
        data: ARRANGEMENT_REQUIREMENTS,
    ): void;
}

const PlatzkarteRequirement: React.FC<IPlatzkarteRequirementProps> = props => {
    const {
        value,
        coachType,
        className,
        passengers,
        requirements,
        coaches,
        schemas,
        onSetRequirements,
    } = props;
    const {isMobile} = useDeviceType();

    const onPlatzkartePlacementChange = useCallback(
        e => {
            const v: ARRANGEMENT_REQUIREMENTS = e.target.value;

            onSetRequirements(setArrangementRequirements, v);
        },
        [onSetRequirements],
    );

    const buttons = useMemo(() => {
        if (coachType !== PLATZKARTE) {
            return null;
        }

        return getButtons(
            countPassengersWithPlaces(passengers),
            requirements,
            coaches,
            schemas,
            coachType,
        );
    }, [coachType, coaches, passengers, requirements, schemas]);

    if (coachType !== PLATZKARTE || !buttons) {
        return null;
    }

    // если у элементов Radiobox поменялось состояние disabled с true на false,
    // то у таких элементов не выставляется состояние hovered при наведении
    // на данный момент в лего нет фикса этого бага, поэтому нужно принудительно перерендерить Radiobox
    const buttonsKey = getButtonsKey(buttons);

    return (
        <div className={cx('root', className)}>
            <Radiobox
                key={buttonsKey}
                onChange={onPlatzkartePlacementChange}
                value={value}
                options={buttons}
                device={isMobile ? 'touch' : 'desktop'}
            />
        </div>
    );
};

export default PlatzkarteRequirement;
