import React, {useCallback, useMemo} from 'react';
import {useDispatch, useSelector} from 'react-redux';
import {PayloadActionCreator, TypeConstant} from 'typesafe-actions';

import {SAPSAN_TRAIN_NAME} from 'projects/trains/constants/rawTrainNames';
import {GENDER_TYPE} from 'projects/trains/constants/genders';

import {
    ITrainDirectionIndexPayload,
    ITrainsCoach,
    ITrainsRequirements,
    TrainsPassengersCount,
} from 'reducers/trains/order/types';
import {ITrainsSchema} from 'server/api/TrainsApi/types/ITrainsDetailsApiResponse';
import {isNotNull} from 'types/utilities';

import currentSegmentDirectionAndIndexSelector from 'selectors/trains/order/currentSegmentDirectionAndIndexSelector';
import currentSegmentSelector from 'selectors/trains/order/currentSegmentSelector';
import currentCoachSelector from 'selectors/trains/order/currentCoachSelector';
import currentTrainDetailsSelector from 'selectors/trains/order/currentTrainDetailsSelector';

import {countPassengersWithPlaces} from 'projects/trains/lib/order/passengers/utils';
import {getThroughCoachArrivalTime} from 'projects/trains/lib/order/getThroughCoachArrivalTime';
import areAllCoachesThroughAndWithSameArrivalTime from 'projects/trains/lib/order/coaches/areAllCoachesThroughAndWithSameArrivalTime';
import getAvailableForSelectGendersForCoaches from 'projects/trains/lib/order/gender/getAvailableForSelectGendersForCoaches';
import needToShowTrainsCoachNumberRequirement from 'projects/trains/components/TrainsOrderPage/Requirements/helpers/needToShowTrainsCoachNumberRequirement';
import isSameISODateString from 'projects/trains/lib/date/isSameISODateString';
import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';
import {useDeviceType} from 'utilities/hooks/useDeviceType';

import * as i18nTrainsTrainDetailsBlock from 'i18n/trains-train-details';
import * as i18nTrainsOrderRestrictionsBlock from 'i18n/trains-order-restrictions';
import * as i18nTrainsCoachPropertiesBlock from 'i18n/trains-coach-properties';
import * as i18nBlock from 'i18n/trains-place-categories';

import StoreyRequirement from 'projects/trains/components/TrainsOrderPage/Requirements/Storey/Storey';
import CoachNumberRequirement from 'projects/trains/components/TrainsOrderPage/Requirements/CoachNumber';
import GenderRequirement from 'projects/trains/components/TrainsOrderPage/Requirements/GenderRequirement/GenderRequirement';
import NearestRequirement from 'projects/trains/components/TrainsOrderPage/Requirements/NearestRequirement/NearestRequirement';
import SapsanRequirements from 'projects/trains/components/TrainsOrderPage/Requirements/SapsanRequirements/SapsanRequirements';
import PlatzkarteRequirement from 'projects/trains/components/TrainsOrderPage/Requirements/PlatzkarteRequirement/PlatzkarteRequirement';
import SingleCompartmentRequirement from 'projects/trains/components/TrainsOrderPage/Requirements/SingleCompartment/SingleCompartmentRequirement';
import ArrangementCountRequirement from 'projects/trains/components/TrainsOrderPage/Requirements/ArrangementCountRequirement/ArrangementCountRequirement';
import MaleIcon from 'icons/16/Male';
import FemaleIcon from 'icons/16/Female';

import cx from './Requirements.scss';

const ICONS = {
    [GENDER_TYPE.MALE]: MaleIcon,
    [GENDER_TYPE.FEMALE]: FemaleIcon,
};

interface IRequirementsProps extends IWithQaAttributes {
    passengers: TrainsPassengersCount;
    coaches: ITrainsCoach[];
    gender: GENDER_TYPE | null;
    schemas: Record<number, ITrainsSchema>;
    trainName: string;
    requirements: ITrainsRequirements;
}

const Requirements: React.FC<IRequirementsProps> = props => {
    const {
        passengers,
        coaches: allCoaches,
        gender,
        trainName,
        schemas,
        requirements,
    } = props;

    const {isMobile} = useDeviceType();

    const dispatch = useDispatch();
    const currentSegmentDirectionAndIndex = useSelector(
        currentSegmentDirectionAndIndexSelector,
    );
    const segment = useSelector(currentSegmentSelector);
    const coach = useSelector(currentCoachSelector);
    const trainDetails = useSelector(currentTrainDetailsSelector);

    const setRequirements = useCallback(
        function <Type extends TypeConstant, P>(
            action: PayloadActionCreator<Type, ITrainDirectionIndexPayload<P>>,
            data: P,
        ) {
            dispatch(action({...currentSegmentDirectionAndIndex, data}));
        },
        [dispatch, currentSegmentDirectionAndIndex],
    );

    // Если в требованиях отображается селект с выбором вагона (кейс с беспересадочными вагонами),
    // то для расчета других требований нужно использовать выбранный вагон.
    const coachesForRequirements = needToShowTrainsCoachNumberRequirement(
        allCoaches,
    )
        ? [coach].filter(isNotNull)
        : allCoaches;

    const genderAttention = useMemo((): React.ReactNode => {
        const gendersForCoaches = getAvailableForSelectGendersForCoaches(
            coachesForRequirements,
            passengers,
        );
        const firstGender = gendersForCoaches[0] as
            | typeof gendersForCoaches[0]
            | undefined;

        if (
            !firstGender ||
            gendersForCoaches.length !== 1 ||
            firstGender === GENDER_TYPE.MIXED ||
            firstGender === GENDER_TYPE.SINGLE
        ) {
            return null;
        }

        const IconComponent = ICONS[firstGender];

        return (
            <div className={cx('genderAttention')}>
                <IconComponent
                    width="15"
                    height="20"
                    className={cx('genderIcon', `genderIcon_${firstGender}`)}
                />

                {i18nBlock.genderDashAttentionDashText({
                    male: firstGender === GENDER_TYPE.MALE,
                })}
            </div>
        );
    }, [coachesForRequirements, passengers]);

    const petsAttention = useMemo((): React.ReactNode => {
        const shouldShowPetsAttention = coachesForRequirements.some(
            ({petsSegregated, petsAllowed}) => petsSegregated && !petsAllowed,
        );

        if (!shouldShowPetsAttention) {
            return null;
        }

        return (
            <div className={cx('petsAttention')}>
                {i18nTrainsCoachPropertiesBlock.petsDashAttentionDashSegregated()}
            </div>
        );
    }, [coachesForRequirements]);

    const throughCoachAttention = useMemo((): React.ReactNode => {
        const firstCoach = allCoaches[0] as typeof allCoaches[0] | undefined;

        if (
            !areAllCoachesThroughAndWithSameArrivalTime(allCoaches) ||
            !firstCoach?.throughArrival ||
            !segment ||
            isSameISODateString(firstCoach.throughArrival, segment.arrival)
        ) {
            return null;
        }

        return (
            <div className={cx('throughCoachAttention')}>
                {i18nBlock.throughDashCoachDashAttentionDashText({
                    count: allCoaches.length,
                    arrival: getThroughCoachArrivalTime({
                        coach: firstCoach,
                        segment: segment,
                        plural: allCoaches.length > 1,
                    }).toLowerCase(),
                })}
            </div>
        );
    }, [allCoaches, segment]);

    const coachesType = allCoaches[0].type;

    const maxPlacesPerCoach = allCoaches.reduce(
        (maxPlaces, {places}) => Math.max(places.length, maxPlaces),
        0,
    );

    if (countPassengersWithPlaces(passengers) > maxPlacesPerCoach) {
        return (
            <div className={cx('root')}>
                <div className={cx('title')}>
                    {i18nTrainsOrderRestrictionsBlock.notDashEnoughDashPlacesDashInDashCoaches()}
                </div>
            </div>
        );
    }

    return (
        <div className={cx('root')} {...prepareQaAttributes(props)}>
            <div className={cx('title')}>
                {i18nTrainsTrainDetailsBlock.chooseDashRequirementsDashForDashComfort()}
            </div>

            {petsAttention}
            {throughCoachAttention}
            {genderAttention}

            <CoachNumberRequirement
                className={cx('coachNumber')}
                segment={segment}
                coach={coach}
                trainDetails={trainDetails}
                coaches={allCoaches}
            />

            <div className={cx('specificRequirements')}>
                <GenderRequirement
                    className={cx('genderTrigger', cx('item'))}
                    coaches={coachesForRequirements}
                    value={gender}
                    passengers={passengers}
                />

                <SingleCompartmentRequirement
                    className={cx('compartmentTrigger', cx('item'))}
                    passengers={passengers}
                    coachType={coachesType}
                    value={requirements.arrangement}
                    coaches={coachesForRequirements}
                    schemas={schemas}
                    requirements={requirements}
                    onSetRequirements={setRequirements}
                />

                <NearestRequirement
                    className={cx('sittingCheckbox', cx('item'))}
                    value={requirements.arrangement}
                    passengers={passengers}
                    coachType={coachesType}
                    trainName={trainName}
                    coaches={coachesForRequirements}
                    schemas={schemas}
                    requirements={requirements}
                    onSetRequirements={setRequirements}
                />

                <PlatzkarteRequirement
                    className={cx('platzkartePlacement', cx('item'))}
                    value={requirements.arrangement}
                    coaches={coachesForRequirements}
                    schemas={schemas}
                    requirements={requirements}
                    coachType={coachesType}
                    passengers={passengers}
                    onSetRequirements={setRequirements}
                />

                {trainName === SAPSAN_TRAIN_NAME && (
                    <SapsanRequirements
                        passengers={passengers}
                        coaches={coachesForRequirements}
                        schemas={schemas}
                        onSetCountRequirements={setRequirements}
                        onSetAdditionalRequirements={setRequirements}
                        isMobile={isMobile}
                    />
                )}
            </div>

            <div className={cx('commonRequirements')}>
                <ArrangementCountRequirement
                    className={cx('placeType', cx('item'))}
                    coaches={coachesForRequirements}
                    schemas={schemas}
                    coachType={coachesType}
                    trainName={trainName}
                    passengers={passengers}
                    requirements={requirements}
                    onClearCountRequirements={setRequirements}
                    onSetCountRequirements={setRequirements}
                />

                <StoreyRequirement
                    className={cx('item')}
                    coaches={coachesForRequirements}
                    schemas={schemas}
                    value={requirements.storey}
                    onSetRequirements={setRequirements}
                />
            </div>
        </div>
    );
};

export default Requirements;
