import React, {Component} from 'react';

import {
    ADDITIONAL_REQUIREMENTS,
    ARRANGEMENT_REQUIREMENTS,
} from 'projects/trains/constants/requirements';

import {
    CountRequirementsType,
    ICompartmentCountRequirements,
    ITrainsCoach,
    TrainsPassengersCount,
} from 'reducers/trains/order/types';
import {ITrainsSchema} from 'server/api/TrainsApi/types/ITrainsDetailsApiResponse';

import {
    setAdditionalRequirements,
    setCountRequirements,
} from 'reducers/trains/order/actions/trains';

import {ALL_SELECT_OPTIONS} from 'projects/trains/components/TrainsOrderPage/Requirements/SapsanRequirements/helpers';
import getAvailableOptionTypes from 'projects/trains/components/TrainsOrderPage/Requirements/SapsanRequirements/helpers/getAvailableOptionTypes';
import getPlacesTypesAndCountsWithDirectionsByCoaches from 'projects/trains/components/TrainsOrderPage/Requirements/SapsanRequirements/helpers/getPlacesTypesAndCountsWithDirectionsByCoaches';
import {countPassengersWithPlaces} from 'projects/trains/lib/order/passengers/utils';
import areAllCoachesWithSchemas from 'projects/trains/lib/order/coaches/areAllCoachesWithSchemas';
import {
    SAPSAN_NEAR_TABLE,
    SAPSAN_NEAR_WINDOW,
    SAPSAN_NEAR_WINDOW_IN_ROW,
    SAPSAN_NO_TABLE,
    SAPSAN_SINGLE,
} from 'projects/trains/lib/order/placesTypes';

import * as i18nBlock from 'i18n/trains-place-categories';

import Checkbox from 'components/Checkbox/Checkbox';
import Select from 'components/Select/Select';

import cx from './SapsanRequirements.scss';

interface ISapsanRequirementsProps {
    coaches: ITrainsCoach[];
    passengers: TrainsPassengersCount;
    schemas: Record<number, ITrainsSchema>;
    onSetCountRequirements(
        actionCreator: typeof setCountRequirements,
        data: CountRequirementsType,
    ): void;
    onSetAdditionalRequirements(
        actionCreator: typeof setAdditionalRequirements,
        data: ADDITIONAL_REQUIREMENTS,
    ): void;
    isMobile: boolean;
}

interface ISapsanRequirementsState {
    selectValue: string;
    checkboxValue: boolean;
    isCheckboxVisible: boolean;
    isCheckboxDisabled: boolean;
}

export default class SapsanRequirements extends Component<
    ISapsanRequirementsProps,
    ISapsanRequirementsState
> {
    state: ISapsanRequirementsState = {
        selectValue: ARRANGEMENT_REQUIREMENTS.IRRELEVANT,
        checkboxValue: false,
        isCheckboxVisible: true,
        isCheckboxDisabled: false,
    };

    componentDidMount(): void {
        if (areAllCoachesWithSchemas(this.props.coaches)) {
            this.setState(this.getStateBySelectValue(this.state.selectValue));
        }
    }

    componentDidUpdate(prevProps: ISapsanRequirementsProps): void {
        if (prevProps.passengers !== this.props.passengers) {
            this.setState(
                this.getStateBySelectValue(ARRANGEMENT_REQUIREMENTS.IRRELEVANT),
            );
        }
    }

    getSelectOptions(): {value: string; data: string}[] {
        const {checkboxValue} = this.state;
        const {coaches, passengers, schemas} = this.props;

        const passengersCount = countPassengersWithPlaces(passengers);

        if (!areAllCoachesWithSchemas(coaches)) {
            return [...ALL_SELECT_OPTIONS]
                .filter(
                    ({val}) =>
                        val !==
                        (passengersCount > 1
                            ? SAPSAN_NEAR_WINDOW
                            : SAPSAN_NEAR_WINDOW_IN_ROW),
                )
                .map(({val: value, text: data}) => ({value, data}));
        }

        const coachesOptionsTypes = getAvailableOptionTypes({
            coaches,
            schemas,
            passengersCount,
            isForward: checkboxValue,
        });

        return [...ALL_SELECT_OPTIONS]
            .filter(({val}) => coachesOptionsTypes.includes(val))
            .map(({val: value, text: data}) => ({value, data}));
    }

    getStateBySelectValue(value: string): ISapsanRequirementsState {
        const {coaches, passengers, schemas} = this.props;

        if (!areAllCoachesWithSchemas(coaches)) {
            return {
                selectValue: value,
                isCheckboxVisible: value !== SAPSAN_NEAR_WINDOW_IN_ROW,
                isCheckboxDisabled: value === SAPSAN_SINGLE,
                checkboxValue: value === SAPSAN_SINGLE,
            };
        }

        const passengersCount = countPassengersWithPlaces(passengers);

        const {maxForwardPlaces} =
            getPlacesTypesAndCountsWithDirectionsByCoaches(coaches, schemas)[
                value
            ];

        return {
            selectValue: value,
            isCheckboxVisible: value !== SAPSAN_NEAR_WINDOW_IN_ROW,
            isCheckboxDisabled:
                value === SAPSAN_SINGLE || maxForwardPlaces < passengersCount,
            checkboxValue:
                maxForwardPlaces >= passengersCount && value === SAPSAN_SINGLE,
        };
    }

    onSelectChange = (value: string): void => {
        this.setState(this.getStateBySelectValue(value), this.setRequirements);
    };

    onCheckboxChange = (e: React.ChangeEvent<HTMLInputElement>): void => {
        this.setState(
            {
                checkboxValue: e.target.checked,
            },
            this.setRequirements,
        );
    };

    setRequirements = (): void => {
        const {selectValue, checkboxValue} = this.state;
        const {passengers} = this.props;

        const countRequirements: ICompartmentCountRequirements = {
            bottom: 0,
            upper: 0,
        };
        let additionalPlaceRequirement: ADDITIONAL_REQUIREMENTS | null = null;

        if (selectValue === SAPSAN_NEAR_WINDOW) {
            if (checkboxValue) {
                additionalPlaceRequirement = ADDITIONAL_REQUIREMENTS.FORWARD;

                // в требованиях к местам для сидячих вагонов bottom - места у окна, upper - у прохода
                countRequirements.bottom =
                    countPassengersWithPlaces(passengers);
            } else {
                additionalPlaceRequirement =
                    ADDITIONAL_REQUIREMENTS.NEAR_WINDOW;
            }
        } else if (selectValue === SAPSAN_NEAR_WINDOW_IN_ROW) {
            additionalPlaceRequirement = ADDITIONAL_REQUIREMENTS.NEAR_WINDOW;
        } else if (selectValue === SAPSAN_NEAR_TABLE) {
            if (checkboxValue) {
                additionalPlaceRequirement =
                    ADDITIONAL_REQUIREMENTS.NEAR_TABLE_AND_FORWARD;
            } else {
                additionalPlaceRequirement = ADDITIONAL_REQUIREMENTS.NEAR_TABLE;
            }
        } else if (selectValue === SAPSAN_NO_TABLE) {
            if (checkboxValue) {
                additionalPlaceRequirement =
                    ADDITIONAL_REQUIREMENTS.NO_TABLE_AND_FORWARD;
            } else {
                additionalPlaceRequirement = ADDITIONAL_REQUIREMENTS.NO_TABLE;
            }
        } else if (selectValue === SAPSAN_SINGLE) {
            additionalPlaceRequirement =
                ADDITIONAL_REQUIREMENTS.SINGLE_AND_FORWARD;
        } else if (checkboxValue) {
            additionalPlaceRequirement = ADDITIONAL_REQUIREMENTS.FORWARD;
        }

        this.props.onSetCountRequirements(
            setCountRequirements,
            countRequirements,
        );

        if (additionalPlaceRequirement) {
            this.props.onSetAdditionalRequirements(
                setAdditionalRequirements,
                additionalPlaceRequirement,
            );
        }
    };

    render(): React.ReactNode {
        const {isMobile} = this.props;
        const {
            checkboxValue,
            isCheckboxVisible,
            isCheckboxDisabled,
            selectValue,
        } = this.state;

        return (
            <div className={cx('root')}>
                <div className={cx('selectWrapper')}>
                    <Select
                        value={selectValue}
                        onChange={this.onSelectChange}
                        options={this.getSelectOptions()}
                        theme="outlined"
                        width="max"
                        size={isMobile ? 'l' : 'm'}
                    />
                </div>

                {isCheckboxVisible && (
                    <Checkbox
                        className={cx('checkbox')}
                        size="m"
                        label={i18nBlock.sapsanDashForward()}
                        checked={checkboxValue}
                        disabled={isCheckboxDisabled}
                        onChange={this.onCheckboxChange}
                    />
                )}
            </div>
        );
    }
}
