import {ITrainsCoachPlace} from 'server/services/TrainsService/types/ITrainsDetailsInfoServiceResponse';
import {ITrainsSchema} from 'server/api/TrainsApi/types/ITrainsDetailsApiResponse';

import isNearWindowPlace from 'projects/trains/components/TrainsOrderPage/Requirements/SapsanRequirements/helpers/isNearWindowPlace';
import getCoachPlacesMap from 'projects/trains/lib/order/coaches/getCoachPlacesMap';

/**
 * Функция для группы мест возвращает количество мест по направлавлению, против направления и признак,
 * есть ли место в группе у окна.
 *
 * @param groupInRow - спискок мест в группе
 * @param placeMap - карта мест
 */
function getForwardAndBackwardPlacesInRow(
    groupInRow: number[],
    placeMap: PartialRecord<number, ITrainsCoachPlace>,
): {
    forward: number;
    backward: number;
    isNearWindow: boolean;
} {
    return groupInRow.reduce<{
        forward: number;
        backward: number;
        isNearWindow: boolean;
    }>(
        (freePlacesInGroupWithDirection, groupPlaceNumber) => {
            const place = placeMap[groupPlaceNumber];

            const type = place?.type;

            if (!type) {
                return freePlacesInGroupWithDirection;
            }

            const isForward = type.indexOf('Forward') !== -1;

            if (isForward) {
                freePlacesInGroupWithDirection.forward++;
            } else {
                freePlacesInGroupWithDirection.backward++;
            }

            if (
                !freePlacesInGroupWithDirection.isNearWindow &&
                isNearWindowPlace(groupPlaceNumber, type)
            ) {
                freePlacesInGroupWithDirection.isNearWindow = true;
            }

            return freePlacesInGroupWithDirection;
        },
        {
            forward: 0,
            backward: 0,
            isNearWindow: false,
        },
    );
}

/**
 * Функция возвращает количество мест "У окна в ряду" для Сапсанов, разделенные по направлению
 *
 * @param places - места
 * @param schema - схема
 */
export default function getNearWindowInRowPlacesWithDirections(
    places: ITrainsCoachPlace[],
    schema: ITrainsSchema | null,
): {forward: number; backward: number} {
    const groupsInRows = schema?.placeFlags?.groupsInRows || [];
    const placeMap = getCoachPlacesMap(places);

    const {forward, backward, notFilledForward, notFilledBackward} =
        groupsInRows.reduce(
            (maxPlacesInRowsWithDirections, groupInRow) => {
                const freePlacesInRow = getForwardAndBackwardPlacesInRow(
                    groupInRow,
                    placeMap,
                );

                const isAllGroupFilled =
                    groupInRow.length ===
                    freePlacesInRow.forward + freePlacesInRow.backward;

                if (freePlacesInRow.isNearWindow) {
                    if (isAllGroupFilled) {
                        maxPlacesInRowsWithDirections.forward +=
                            freePlacesInRow.forward;
                        maxPlacesInRowsWithDirections.backward +=
                            freePlacesInRow.backward;
                    } else {
                        maxPlacesInRowsWithDirections.notFilledForward =
                            Math.max(
                                maxPlacesInRowsWithDirections.notFilledForward,
                                freePlacesInRow.forward,
                            );
                        maxPlacesInRowsWithDirections.notFilledBackward =
                            Math.max(
                                maxPlacesInRowsWithDirections.notFilledBackward,
                                freePlacesInRow.backward,
                            );
                    }
                }

                return maxPlacesInRowsWithDirections;
            },
            {
                forward: 0,
                backward: 0,
                notFilledForward: 0,
                notFilledBackward: 0,
            },
        );

    return {
        forward: forward + notFilledForward,
        backward: backward + notFilledBackward,
    };
}
