import {ITrainsCoach} from 'reducers/trains/order/types';
import {ITrainsSchema} from 'server/api/TrainsApi/types/ITrainsDetailsApiResponse';

import getPlacesTypesAndCountsWithDirections from 'projects/trains/components/TrainsOrderPage/Requirements/SapsanRequirements/helpers/getPlacesTypesAndCountsWithDirections';
import getSchemaByCoach from 'projects/trains/lib/order/getSchemaByCoach';

/**
 * Функция добавляет свойство maxForwardPlaces для дальнейшей работы
 *
 * @param coachPlacesTypesAndCounts - количество мест по типам и направлениям для вагона
 */
function addMaxForwardPlacesProp(
    coachPlacesTypesAndCounts: Record<
        string,
        {forward: number; backward: number}
    >,
): Record<
    string,
    {forward: number; backward: number; maxForwardPlaces: number}
> {
    return Object.entries(coachPlacesTypesAndCounts).reduce<
        Record<
            string,
            {forward: number; backward: number; maxForwardPlaces: number}
        >
    >((accPlaceCountsByDirection, [type, placeCountsByDirection]) => {
        accPlaceCountsByDirection[type] = {
            ...placeCountsByDirection,
            maxForwardPlaces: placeCountsByDirection.forward,
        };

        return accPlaceCountsByDirection;
    }, {});
}

/**
 * Функция для всех вагонов возвращает максимальное количество мест по каждому типу
 *
 * @param coaches - вагоны
 * @param schemas - схемы
 */
export default function getPlacesTypesAndCountsWithDirectionsByCoaches(
    coaches: ITrainsCoach[],
    schemas: Record<number, ITrainsSchema>,
): Record<
    string,
    {forward: number; backward: number; maxForwardPlaces: number}
> {
    return coaches.reduce<
        Record<
            string,
            {forward: number; backward: number; maxForwardPlaces: number}
        >
    >((maxPlacesTypesAndCountWithDirection, coach) => {
        const {places} = coach;
        const coachPlacesTypesAndCounts = addMaxForwardPlacesProp(
            getPlacesTypesAndCountsWithDirections(
                places,
                getSchemaByCoach(coach, schemas),
            ),
        );

        if (!maxPlacesTypesAndCountWithDirection) {
            return coachPlacesTypesAndCounts;
        }

        Object.entries(coachPlacesTypesAndCounts).forEach(
            ([placeType, placeCountsByDirection]) => {
                if (!maxPlacesTypesAndCountWithDirection[placeType]) {
                    maxPlacesTypesAndCountWithDirection[placeType] =
                        placeCountsByDirection;

                    return;
                }

                const typePlacesWithDirectionsByCoaches =
                    maxPlacesTypesAndCountWithDirection[placeType];

                const {forward, backward, maxForwardPlaces} =
                    placeCountsByDirection;

                typePlacesWithDirectionsByCoaches.maxForwardPlaces = Math.max(
                    typePlacesWithDirectionsByCoaches.maxForwardPlaces,
                    maxForwardPlaces,
                );

                if (
                    forward + backward >
                    typePlacesWithDirectionsByCoaches.forward +
                        typePlacesWithDirectionsByCoaches.backward
                ) {
                    typePlacesWithDirectionsByCoaches.forward = forward;
                    typePlacesWithDirectionsByCoaches.backward = backward;
                }
            },
        );

        return maxPlacesTypesAndCountWithDirection;
    }, {});
}
