import React, {ChangeEvent, useCallback, useMemo} from 'react';

import {ARRANGEMENT_REQUIREMENTS} from 'projects/trains/constants/requirements';
import {COMPARTMENT} from 'projects/trains/lib/segments/tariffs/constants';
import {TRAIN_COACH_TYPE} from 'projects/trains/constants/coachType';

import {IWithClassName} from 'types/withClassName';
import {
    IMappedCoach,
    ITrainsRequirements,
    TrainsPassengersCount,
} from 'reducers/trains/order/types';
import {ITrainsSchema} from 'server/api/TrainsApi/types/ITrainsDetailsApiResponse';

import {setArrangementRequirements} from 'reducers/trains/order/actions/trains';

import isArrangementOptionAvailable from 'projects/trains/components/TrainsOrderPage/Requirements/helpers/isArrangementOptionAvailable';
import {countPassengersWithPlaces} from 'projects/trains/lib/order/passengers/utils';
import {useDeviceType} from 'utilities/hooks/useDeviceType';

import * as i18nBlock from 'i18n/trains-place-categories';

import RadioButton from 'components/RadioButton/RadioButton';

import cx from './SingleCompartmentRequirement.scss';

export interface ISingleCompartmentRequirementProps extends IWithClassName {
    value?: ARRANGEMENT_REQUIREMENTS;
    requirements: ITrainsRequirements;
    coaches: IMappedCoach[];
    schemas: Record<number, ITrainsSchema>;
    passengers: TrainsPassengersCount;
    coachType: TRAIN_COACH_TYPE;
    onSetRequirements: (
        actionCreator: typeof setArrangementRequirements,
        data: ARRANGEMENT_REQUIREMENTS,
    ) => void;
}

const SingleCompartmentRequirement: React.FC<ISingleCompartmentRequirementProps> =
    ({
        value,
        requirements,
        coaches,
        schemas,
        passengers,
        coachType,
        className,
        onSetRequirements,
    }) => {
        const deviceType = useDeviceType();

        const passengersCount = countPassengersWithPlaces(passengers);
        const buttons: {
            value: ARRANGEMENT_REQUIREMENTS;
            label: string;
            disabled: boolean;
        }[] = useMemo(
            () => [
                {
                    value: ARRANGEMENT_REQUIREMENTS.COMPARTMENT,
                    label: i18nBlock.requirementDashCompartment(),
                    disabled: !isArrangementOptionAvailable({
                        option: ARRANGEMENT_REQUIREMENTS.COMPARTMENT,
                        requirements,
                        coaches,
                        schemas,
                        passengersCount,
                        coachType,
                    }),
                },
                {
                    value: ARRANGEMENT_REQUIREMENTS.IRRELEVANT,
                    label: i18nBlock.requirementDashIrrelevant(),
                    disabled: false,
                },
            ],
            [requirements, coaches, schemas, passengersCount, coachType],
        );
        const handleChange = useCallback(
            (event: ChangeEvent<HTMLInputElement>) => {
                const value = event.target.value as ARRANGEMENT_REQUIREMENTS;

                onSetRequirements(setArrangementRequirements, value);
            },
            [onSetRequirements],
        );

        if (coachType !== COMPARTMENT || passengersCount === 1) {
            return null;
        }

        return (
            <RadioButton
                className={cx('root', className)}
                name="singleCompartmentRequirement"
                value={value}
                size={deviceType.isMobile ? 'l' : 'm-inset'}
                onChange={handleChange}
                width={deviceType.isMobile ? 'max' : 'auto'}
            >
                {buttons.map(({label, value: buttonValue, disabled}) => (
                    <RadioButton.Radio
                        key={label}
                        value={buttonValue}
                        checked={value === buttonValue}
                        disabled={disabled}
                    >
                        {label}
                    </RadioButton.Radio>
                ))}
            </RadioButton>
        );
    };

export default SingleCompartmentRequirement;
