import React, {ChangeEvent, useCallback, useMemo} from 'react';

import {REQUIREMENTS_IRRELEVANT_OPTION} from 'projects/trains/constants/requirements';

import {IWithClassName} from 'types/withClassName';
import {
    IMappedCoach,
    StoreyRequirementsType,
} from 'reducers/trains/order/types';
import {ITrainsSchema} from 'server/api/TrainsApi/types/ITrainsDetailsApiResponse';

import {setStoreyRequirements} from 'reducers/trains/order/actions/trains';

import getSchemaByCoach from 'projects/trains/lib/order/getSchemaByCoach';
import {useDeviceType} from 'utilities/hooks/useDeviceType';

import * as i18nTrainsPlaceCategoriesBlock from 'i18n/trains-place-categories';
import * as i18nBlock from 'i18n/trains-transport-schema';

import RadioButton from 'components/RadioButton/RadioButton';

import cx from './Storey.scss';

interface IStoreyRadioButton {
    value: StoreyRequirementsType;
    label: string;
    disabled?: boolean;
}

const getButtons = (
    coach: IMappedCoach,
    schema: ITrainsSchema | null,
): IStoreyRadioButton[] => {
    const {places = []} = coach;
    const {secondStorey = []} = schema?.placeFlags || {};
    const informationAboutFloorsExists = Boolean(
        places.length && secondStorey.length,
    );

    const secondFloorFirstPlace = secondStorey[0];
    const secondFloorPlaces = places.filter(
        ({number}) => secondFloorFirstPlace && number >= secondFloorFirstPlace,
    );
    const firstFloorIsOutOfSell =
        informationAboutFloorsExists &&
        places.length === secondFloorPlaces.length;
    const secondFloorIsOutOfSell =
        informationAboutFloorsExists && !secondFloorPlaces.length;

    return [
        {
            value: 1,
            disabled: firstFloorIsOutOfSell,
            label: `1 ${i18nBlock.coachDashStorey()}`,
        },
        {
            value: 2,
            disabled: secondFloorIsOutOfSell,
            label: `2 ${i18nBlock.coachDashStorey()}`,
        },
        {
            value: REQUIREMENTS_IRRELEVANT_OPTION,
            label: i18nTrainsPlaceCategoriesBlock.requirementDashIrrelevant(),
        },
    ];
};

export interface IStoreyRequirementProps extends IWithClassName {
    value?: StoreyRequirementsType;
    coaches: IMappedCoach[];
    schemas: Record<number, ITrainsSchema>;
    onSetRequirements: (
        actionCreator: typeof setStoreyRequirements,
        data: StoreyRequirementsType,
    ) => void;
}

const StoreyRequirement: React.FC<IStoreyRequirementProps> = ({
    value,
    coaches,
    schemas,
    className,
    onSetRequirements,
}) => {
    const deviceType = useDeviceType();

    const handleChange = useCallback(
        (event: ChangeEvent<HTMLInputElement>) => {
            const storey = parseInt(event.target.value, 10);

            onSetRequirements(
                setStoreyRequirements,
                isNaN(storey)
                    ? (event.target.value as StoreyRequirementsType)
                    : storey,
            );
        },
        [onSetRequirements],
    );

    const allCoachesButtons = useMemo(
        () =>
            coaches.reduce((buttons: IStoreyRadioButton[] | null, coach) => {
                const schema = getSchemaByCoach(coach, schemas);

                if (!buttons) {
                    return getButtons(coach, schema);
                }

                const coachButtons = getButtons(coach, schema).slice(0, 2);

                coachButtons.forEach((button, index) => {
                    if (!button.disabled) {
                        buttons[index].disabled = false;
                    }
                });

                return buttons;
            }, null),
        [coaches, schemas],
    );

    if (coaches.every(coach => !coach.twoStorey) || !allCoachesButtons) {
        return null;
    }

    return (
        <RadioButton
            className={cx('root', className)}
            name="storeyRequirement"
            value={value}
            size={deviceType.isMobile ? 'l' : 'm-inset'}
            onChange={handleChange}
            width={deviceType.isMobile ? 'max' : 'auto'}
        >
            {allCoachesButtons.map(({label, value: buttonValue, disabled}) => (
                <RadioButton.Radio
                    key={label}
                    value={buttonValue}
                    checked={value === buttonValue}
                    disabled={disabled}
                >
                    {label}
                </RadioButton.Radio>
            ))}
        </RadioButton>
    );
};

export default StoreyRequirement;
