import {
    ARRANGEMENT_REQUIREMENTS,
    MAX_PLACE_TYPE_COUNT,
} from 'projects/trains/constants/requirements';
import {PLATZKARTE} from 'projects/trains/lib/segments/tariffs/constants';
import {TRAIN_COACH_TYPE} from 'projects/trains/constants/coachType';

import {ITrainsCoach, ITrainsRequirements} from 'reducers/trains/order/types';
import {ITrainsSchema} from 'server/api/TrainsApi/types/ITrainsDetailsApiResponse';

import getUpperAndBottomPlacesCounts from 'projects/trains/components/TrainsOrderPage/Requirements/helpers/getUpperAndBottomPlacesCounts';
import getPlacesGroupedByBlocks from 'projects/trains/components/TrainsOrderPage/Requirements/helpers/getPlacesGroupedByBlocks';
import getCoachPlacesMap from 'projects/trains/lib/order/coaches/getCoachPlacesMap';
import getSchemaByCoach from 'projects/trains/lib/order/getSchemaByCoach';
import areAllCoachesWithSchemas from 'projects/trains/lib/order/coaches/areAllCoachesWithSchemas';

/**
 * Функция проверяет, доступна ли опция с выбором типа расположения мест при уже заданных требованиях.
 *
 * @param params
 * @param params.option - опция с количеством верхних и нижних мест
 * @param params.coaches - вагоны
 * @param params.schemas - схемы
 * @param params.passengersCount - количество пассажиров
 * @param params.coachType - тип вагона
 * @param params.requirements - требования к местам
 */
export default function isArrangementOptionAvailable({
    option,
    coaches,
    schemas,
    passengersCount,
    coachType,
    requirements,
}: {
    option: ARRANGEMENT_REQUIREMENTS;
    coaches: ITrainsCoach[];
    schemas: Record<number, ITrainsSchema>;
    passengersCount: number;
    coachType: TRAIN_COACH_TYPE;
    requirements: ITrainsRequirements;
}): boolean {
    const count = requirements.count;
    const upper = (count && 'upper' in count && count.upper) || 0;
    const bottom = (count && 'bottom' in count && count.bottom) || 0;
    const maxCountOfType = Math.max(upper, bottom);

    if (
        (option === ARRANGEMENT_REQUIREMENTS.COMPARTMENT ||
            option === ARRANGEMENT_REQUIREMENTS.SECTION) &&
        (passengersCount === 1 ||
            maxCountOfType > MAX_PLACE_TYPE_COUNT[PLATZKARTE][option])
    ) {
        return false;
    }

    if (!areAllCoachesWithSchemas(coaches)) {
        return true;
    }

    return coaches.some(coach => {
        const schema = getSchemaByCoach(coach, schemas);
        const placeFlags = schema?.placeFlags || null;
        const upperPlaces = placeFlags?.upper || [];

        if (!placeFlags) {
            return false;
        }

        const blocksOfPlaces = getPlacesGroupedByBlocks({
            placesType: option,
            coachType,
            placeFlags,
            coach,
        });
        const placeMap = getCoachPlacesMap(coach.places);

        return blocksOfPlaces.some(blockPlaces => {
            const placesByType = getUpperAndBottomPlacesCounts(
                placeMap,
                blockPlaces,
                upperPlaces,
            );

            if (upper || bottom) {
                return (
                    placesByType.upper >= upper && placesByType.bottom >= bottom
                );
            }

            return placesByType.upper + placesByType.bottom >= passengersCount;
        });
    });
}
