import {TRAIN_COACH_TYPE} from 'projects/trains/constants/coachType';

import {ITrainsCoach, ITrainsRequirements} from 'reducers/trains/order/types';
import {ITrainsSchema} from 'server/api/TrainsApi/types/ITrainsDetailsApiResponse';

import getPlacesGroupedByBlocks from 'projects/trains/components/TrainsOrderPage/Requirements/helpers/getPlacesGroupedByBlocks';
import getUpperAndBottomPlacesCounts from 'projects/trains/components/TrainsOrderPage/Requirements/helpers/getUpperAndBottomPlacesCounts';
import getSchemaByCoach from 'projects/trains/lib/order/getSchemaByCoach';
import getCoachPlacesMap from 'projects/trains/lib/order/coaches/getCoachPlacesMap';

/**
 * Функция проверяет, доступна ли опция с выбором количества верхних и нижних мест при уже заданных требованиях.
 *
 * @param option - опция с количеством верхних и нижних мест
 * @param coaches - вагоны
 * @param schemas - схемы
 * @param passengersCount - количество пассажиров
 * @param coachType - тип вагона
 * @param requirements - требования к местам
 */
export default function isUpperAndBottomCountsOptionAvailable({
    option,
    coaches,
    schemas,
    passengersCount,
    coachType,
    requirements,
}: {
    option: [number, number];
    coaches: ITrainsCoach[];
    schemas: Record<number, ITrainsSchema>;
    passengersCount: number;
    coachType: TRAIN_COACH_TYPE;
    requirements: ITrainsRequirements;
}): boolean {
    const [upper, bottom] = option;

    return coaches.some(coach => {
        const schema = getSchemaByCoach(coach, schemas);
        const placeFlags = schema?.placeFlags;

        if (!placeFlags) {
            return false;
        }

        const upperPlaces = placeFlags.upper || [];
        const blocksOfPlaces = getPlacesGroupedByBlocks({
            placesType: requirements.arrangement,
            coachType,
            placeFlags,
            coach,
        });
        const placeMap = getCoachPlacesMap(coach.places);

        return blocksOfPlaces.some(blockPlaces => {
            const placesByType = getUpperAndBottomPlacesCounts(
                placeMap,
                blockPlaces,
                upperPlaces,
            );

            if (upper || bottom) {
                return (
                    placesByType.upper >= upper && placesByType.bottom >= bottom
                );
            }

            return placesByType.upper + placesByType.bottom >= passengersCount;
        });
    });
}
