import React, {useCallback, useEffect, useRef, useState} from 'react';
import {subscribe, unsubscribe} from 'pubsub-js';

import {ITrainsCoach, TrainsPassengersCount} from 'reducers/trains/order/types';
import {ITrainsSchema} from 'server/api/TrainsApi/types/ITrainsDetailsApiResponse';
import {EPubSubEvent} from 'types/EPubSubEvent';

import {countPassengersWithPlaces} from 'projects/trains/lib/order/passengers/utils';
import getCoachPlacesPubsubEventName from 'projects/trains/lib/order/coaches/getCoachPlacesPubsubEventName';
import {IPlaceMapItem} from 'projects/trains/lib/order/getPlaceMap';
import getTooltipText from './utilities/getTooltipText';
import {getOffset} from './utilities/getOffset';
import {useBoolean} from 'utilities/hooks/useBoolean';
import {isTooltipAvailable} from './utilities/isTooltipAvailable';

import {ISchemaPlaceCoords} from 'projects/trains/components/TrainsOrderPage/SvgSchema/SvgSchema';

import cx from './SchemeTooltip.scss';

interface ISchemeTooltipProps {
    coach: ITrainsCoach;
    schema: ITrainsSchema;
    placeMap: IPlaceMapItem[];
    passengers: TrainsPassengersCount;
}

interface ISchemeTooltipState {
    content: React.ReactNode;
    upper: boolean;
    style: Pick<React.CSSProperties, 'top' | 'left'>;
}

const SchemeTooltip: React.FC<ISchemeTooltipProps> = props => {
    const {coach, schema, placeMap, passengers} = props;

    const [state, setState] = useState<ISchemeTooltipState>({
        content: '',
        upper: false,
        style: {
            top: 0,
            left: 0,
        },
    });
    const {value: isVisible, setFalse: hide, setTrue: show} = useBoolean(false);

    const pubSubEvents = useRef<PartialRecord<EPubSubEvent, string>>({});

    const handleMouseAction = useCallback(
        (
            _event: EPubSubEvent,
            {place, coords}: {place: IPlaceMapItem; coords: ISchemaPlaceCoords},
        ): void => {
            const {upper} = place;
            const passengersCount = countPassengersWithPlaces(passengers);

            if (!isTooltipAvailable(place, schema)) {
                hide();

                return;
            }

            const content = getTooltipText({place, passengersCount, placeMap});

            setState({
                style: getOffset(place, coords),
                upper,
                content,
            });
            show();
        },
        [hide, passengers, placeMap, schema, show],
    );

    useEffect(() => {
        pubSubEvents.current = {
            [EPubSubEvent.SCHEME_MOUSE_MOVE]: subscribe(
                getCoachPlacesPubsubEventName(
                    EPubSubEvent.SCHEME_MOUSE_MOVE,
                    coach,
                ),
                handleMouseAction,
            ),
            [EPubSubEvent.SCHEME_PLACE_ENTER]: subscribe(
                getCoachPlacesPubsubEventName(
                    EPubSubEvent.SCHEME_PLACE_ENTER,
                    coach,
                ),
                handleMouseAction,
            ),
            [EPubSubEvent.SCHEME_PLACE_LEAVE]: subscribe(
                getCoachPlacesPubsubEventName(
                    EPubSubEvent.SCHEME_PLACE_LEAVE,
                    coach,
                ),
                hide,
            ),
            [EPubSubEvent.SCHEME_CLOSE_ALL]: subscribe(
                EPubSubEvent.SCHEME_CLOSE_ALL,
                hide,
            ),
        };

        return () => {
            Object.values(pubSubEvents.current).forEach(unsubscribe);

            pubSubEvents.current = {};
        };
    }, [coach, handleMouseAction, hide]);

    const {style, upper, content} = state;

    return (
        <span
            className={cx('root', {
                root_upper: upper,
                root_visible: isVisible,
            })}
            style={style}
        >
            {content}
        </span>
    );
};

export default SchemeTooltip;
