import React, {useCallback, useMemo} from 'react';
import {useDispatch, useSelector} from 'react-redux';

import {TRAIN_COACH_TYPE} from 'projects/trains/constants/coachType';

import {ITrainsCoach, ITrainsDetails} from 'reducers/trains/order/types';
import {ICoachesTypeGroup} from 'projects/trains/components/TrainsOrderPage/SimpleSelectorView/types/TCoachesTypeGroups';
import {ETrainsGoal} from 'utilities/metrika/types/goals/trains';

import {changeCoachType} from 'reducers/trains/order/thunk/changeCoachType';
import {setExpandedServiceClass} from 'reducers/trains/order/actions/trains';
import {changeCoachAndResetRequirements} from 'reducers/trains/actions/changeCoachAndResetRequirements';

import passengersCountSelector from 'selectors/trains/order/passengersCountSelector';
import currentCoachTypeSelector from 'selectors/trains/order/currentCoachTypeSelector';
import currentExpandedServiceClassSelector from 'selectors/trains/order/currentExpandedServiceClassSelector';
import currentSegmentDirectionAndIndexSelector from 'selectors/trains/order/currentSegmentDirectionAndIndexSelector';

import {reachGoalOnce} from 'utilities/metrika';
import {deviceMods} from 'utilities/stylesUtils';
import {useDeviceType} from 'utilities/hooks/useDeviceType';
import {useExperiments} from 'utilities/hooks/useExperiments';
import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';
import {calculateCoachTabsOptions} from './utilities/calculateCoachTabsOptions';
import {countPassengersWithPlaces} from 'projects/trains/lib/order/passengers/utils';
import getCoachesGroupedByTypeWithInfo from 'projects/trains/lib/order/coaches/getCoachesGroupedByTypeWithInfo';

import {chooseDashCoachDashTypeDashAndDashPlaces} from 'i18n/trains-train-details';

import BookingLayout from 'components/Layouts/BookingLayout/BookingLayout';
import Coaches from 'projects/trains/components/TrainsOrderPage/Coaches/Coaches';
import CoachTypeSelector from 'projects/trains/components/TrainsOrderPage/SimpleSelectorView/components/CoachTypeSelector/CoachTypeSelector';

import {useChangeActiveGroup} from './hooks/useChangeActiveGroup';

import cx from './SimpleSelectorView.scss';

interface ISimpleSelectorViewProps extends IWithQaAttributes {
    allCoaches: ITrainsCoach[];
    coachesWithEnoughPlaces: ITrainsCoach[];
    forceClassSelector?: boolean;
    trainDetails?: Nullable<ITrainsDetails>;
}

const SimpleSelectorView: React.FC<ISimpleSelectorViewProps> = props => {
    const {
        allCoaches,
        trainDetails,
        coachesWithEnoughPlaces,
        forceClassSelector,
    } = props;
    const dispatch = useDispatch();
    const deviceType = useDeviceType();
    const {isDesktop} = deviceType;
    const {trainsDefaultActiveTab} = useExperiments();
    const trainsDisableCoachTypeTabs = false;
    const currentDirectionAndIndex = useSelector(
        currentSegmentDirectionAndIndexSelector,
    );
    const coachType = useSelector(currentCoachTypeSelector);
    const expandedServiceClass = useSelector(
        currentExpandedServiceClassSelector,
    );
    const passengers = useSelector(passengersCountSelector);

    const {groups, activeGroup, activeClass} = useMemo(() => {
        if (!trainDetails) {
            return {
                activeGroup: null,
                activeClass: null,
                groups: [] as ICoachesTypeGroup[],
            };
        }

        const groupsByType = getCoachesGroupedByTypeWithInfo({
            coaches: trainsDisableCoachTypeTabs
                ? allCoaches
                : coachesWithEnoughPlaces,
            schemas: trainDetails.schemas,
            beddingOption: true,
        });

        const aGroup = groupsByType.find(({type}) => type === coachType);
        const aClass = aGroup?.classes.find(
            ({key}) => key === expandedServiceClass,
        );

        return {
            groups: groupsByType,
            activeGroup: aGroup,
            activeClass: aClass,
        };
    }, [
        trainDetails,
        allCoaches,
        coachesWithEnoughPlaces,
        coachType,
        expandedServiceClass,
        trainsDisableCoachTypeTabs,
    ]);

    const isClassSelector = groups.length === 1 && forceClassSelector;

    const options = useMemo(() => {
        return calculateCoachTabsOptions({
            groups,
            passengers,
            activeGroup,
            isClassSelector,
            trainsDisableCoachTypeTabs,
        });
    }, [
        groups,
        activeGroup,
        isClassSelector,
        passengers,
        trainsDisableCoachTypeTabs,
    ]);

    const handleClassChange = useCallback(
        (value: string) => {
            if (value === activeClass?.key) {
                return;
            }

            dispatch(
                setExpandedServiceClass({
                    ...currentDirectionAndIndex,
                    data: value,
                }),
            );

            if (!activeGroup) {
                return;
            }

            const nextActiveClass = activeGroup.classes.find(
                item => item.key === value,
            );

            if (!nextActiveClass) {
                return;
            }

            dispatch(
                changeCoachAndResetRequirements(nextActiveClass.coaches[0]),
            );
        },
        [dispatch, currentDirectionAndIndex, activeGroup, activeClass],
    );
    const handleTypeChange = useCallback(
        (value: TRAIN_COACH_TYPE) => {
            if (activeGroup?.type === value) {
                return;
            }

            reachGoalOnce(ETrainsGoal.COACH_TYPE_WAS_CHANGED);
            dispatch(changeCoachType(value));
        },
        [dispatch, activeGroup],
    );

    const coaches = useMemo(
        () =>
            isClassSelector
                ? activeClass?.coaches || []
                : activeGroup?.coaches || [],
        [isClassSelector, activeClass, activeGroup],
    );

    const selectorValue = isClassSelector
        ? activeClass?.key
        : activeGroup?.type;
    const selectorHandler = isClassSelector
        ? handleClassChange
        : handleTypeChange;

    useChangeActiveGroup({
        activeGroup,
        options,
        handleClassChange,
        handleTypeChange,
        selectorValue,
        trainsDefaultActiveTab,
    });

    return (
        <BookingLayout.Forms
            className={cx('root', deviceMods('root', deviceType))}
        >
            <BookingLayout.Card>
                <h2 className={cx('title', deviceMods('title', deviceType))}>
                    {chooseDashCoachDashTypeDashAndDashPlaces({
                        count: countPassengersWithPlaces(passengers),
                    })}
                </h2>
                {activeGroup && (
                    <CoachTypeSelector
                        className={cx('typeSelector')}
                        value={selectorValue}
                        options={options}
                        onChange={selectorHandler}
                        size={isDesktop ? 'l' : 'm'}
                        {...prepareQaAttributes({
                            parent: props,
                            current: 'coachTypeTabsSelector',
                        })}
                    />
                )}
                <Coaches
                    coaches={coaches}
                    showCoachNumbers
                    {...prepareQaAttributes({
                        parent: props,
                        current: 'coaches',
                    })}
                />
            </BookingLayout.Card>
        </BookingLayout.Forms>
    );
};

export default SimpleSelectorView;
